<?php
/**
 * @package   DPCalendar
 * @copyright Copyright (C) 2014 Digital Peak GmbH. <https://www.digital-peak.com>
 * @license   https://www.gnu.org/licenses/gpl-3.0.html GNU/GPL
 */

namespace DigitalPeak\Plugin\DPCalendar\JEvents\Extension;

\defined('_JEXEC') or die();

use DigitalPeak\Component\DPCalendar\Administrator\Helper\DPCalendarHelper;
use DigitalPeak\Component\DPCalendar\Administrator\Plugin\DPCalendarPlugin;
use Joomla\CMS\Categories\Categories;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Date\Date;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Registry\Registry;
use Joomla\Utilities\ArrayHelper;

// @phpstan-ignore-next-line
\JLoader::register('JevRegistry', JPATH_SITE . '/components/com_jevents/libraries/registry.php');

class JEvents extends DPCalendarPlugin
{
	use DatabaseAwareTrait;

	protected string $identifier = 'je';

	protected function getContent(string $calendarId, ?Date $startDate = null, ?Date $endDate = null, ?Registry $options = null): string
	{
		if (!is_dir(JPATH_ADMINISTRATOR . '/components/com_jevents')) {
			return '';
		}

		if ($startDate == null) {
			// Default to now
			$startDate = DPCalendarHelper::getDate();
		}
		if ($endDate == null) {
			// Default to max end
			$endDate = DPCalendarHelper::getDate(date('Y-m-d H:i:s', PHP_INT_MAX));
		}

		$db    = $this->getDatabase();
		$query = $db->getQuery(true);
		$query->select('DISTINCT e.*, d.*, e.rawdata as ical_data');
		$query->from('#__jevents_vevent e');
		$query->join('LEFT', '#__jevents_vevdetail d ON d.evdet_id = e.detail_id');

		$query->where(($this->params->get('use_categories') ? 'e.catid' : 'e.icsid') . ' = ' . (int)$calendarId);

		$subQuery = $db->getQuery(true);
		$subQuery->select('eventid, min(startrepeat) as min_date, max(endrepeat) as max_date');
		$subQuery->from('#__jevents_repetition');
		$subQuery->group('eventid');

		$query->join('RIGHT', '(' . $subQuery . ') r ON r.eventid = e.detail_id');
		$query->where(
			'(r.min_date between ' . $db->quote($startDate->toSql(false)) . ' and ' . $db->quote($endDate->toSql(false)) .
			// Allow if max date is between range
			' or r.max_date between ' . $db->quote($startDate->toSql(false)) . ' and ' . $db->quote($endDate->toSql(false)) .
			// Allow if range is between min and max
			' or (r.min_date < ' . $db->quote($startDate->toSql(false)) . ' and r.max_date > ' . $db->quote($endDate->toSql(false)) . ')' .
			')'
		);

		$db->setQuery($query);
		$rows = $db->loadObjectList();

		$compParams      = ComponentHelper::getParams('com_jevents');
		$jeventsTimezone = $compParams->get('icaltimezonelive');
		if ($jeventsTimezone) {
			$jeventsTimezone = new \DateTimeZone($jeventsTimezone);
		}

		$text   = [];
		$text[] = 'BEGIN:VCALENDAR';
		foreach ($rows as $event) {
			$data = @unserialize($event->ical_data);
			if (!\is_array($data)) {
				$this->log('Ical Data could not being parsed of the event ' . $event->summary);
				continue;
			}

			$text[] = 'BEGIN:VEVENT';
			$text[] = 'X-ACCESS:' . $event->access;
			$text[] = 'X-MODIFIED:' . $event->modified;
			$text[] = 'X-HITS:' . $event->hits;
			$text[] = 'X-URL:' . $event->url;
			foreach ($data as $key => $value) {
				switch (strtolower((string)$key)) {
					case 'dtstart':
						if ($data['allDayEvent'] == 'on') {
							$key   = 'DTSTART;VALUE=DATE';
							$value = DPCalendarHelper::getDate($value, true);
							if ($jeventsTimezone) {
								$value->setTimezone($jeventsTimezone);
							}
							$value = $value->format('Ymd', true);
						} else {
							$value = DPCalendarHelper::getDate($value, false)->format('Ymd\THis\Z');
						}
						break;
					case 'dtend':
						if ($data['allDayEvent'] == 'on') {
							$key   = 'DTEND;VALUE=DATE';
							$value = DPCalendarHelper::getDate($value + 86400, true);
							if ($jeventsTimezone) {
								$value->setTimezone($jeventsTimezone);
							}
							$value = $value->format('Ymd', true);
						} else {
							$value = DPCalendarHelper::getDate($value, false)->format('Ymd\THis\Z');
						}
						break;
					case 'rrule':
						$tmp = '';
						foreach ($value as $rKey => $rValue) {
							if (\is_array($rValue)) {
								continue;
							}

							if ($rKey == 'BYYEARDAY') {
								$rValue = str_replace('+', '', (string)$rValue);
							}

							if ($rKey == 'UNTIL') {
								$rValue = DPCalendarHelper::getDate($rValue + 86400, false)->format('Ymd\T000000\Z');
							}
							$tmp .= $rKey . '=' . $rValue . ';';
						}
						$value = trim($tmp, ';');
						if (stripos($value, 'FREQ=NONE') !== false) {
							$key = null;
						}
						break;
					case 'description':
						$value = $this->getDPCalendar()->getMVCFactory()->createModel('Ical', 'Administrator')->icalEncode($this->replaceNl($value));
						break;
					case 'publish_down':
					case 'publish_down2':
					case 'publish_up':
					case 'publish_up2':
					case 'alldayevent':
					case 'locakevent':
					case 'x-extrainfo':
						$key = null;
						break;
				}
				if ($key === null) {
					continue;
				}
				$text[] = $key . ':' . $value;
			}

			$text[] = 'END:VEVENT';
		}
		$text[] = 'END:VCALENDAR';

		return implode(PHP_EOL, $text);
	}

	/**
	 * @return mixed[]
	 */
	protected function fetchCalendars(array $calendarIds = []): array
	{
		if (!is_dir(JPATH_ADMINISTRATOR . '/components/com_jevents')) {
			return [];
		}

		if ($this->params->get('use_categories')) {
			// @phpstan-ignore-next-line
			$calendar = Categories::getInstance('JEvents');
			if (!$calendar instanceof Categories) {
				return [];
			}

			$root = $calendar->get('root');
			if ($root === null) {
				return [];
			}

			$calendars = [];
			foreach ($root->getChildren(true) as $category) {
				if ($calendarIds !== [] && !\in_array($category->id, $calendarIds)) {
					continue;
				}
				$calendars[] = $this->createCalendar((string)$category->id, $category->title, $category->description);
			}

			return $calendars;
		}

		$query = 'select * from #__jevents_icsfile';
		if ($calendarIds !== []) {
			$query .= ' where ics_id IN (' . implode(',', ArrayHelper::toInteger($calendarIds)) . ')';
		}
		$db = $this->getDatabase();
		$db->setQuery($query);

		$rows = $db->loadObjectList();

		$calendars = [];
		foreach ($rows as $calendar) {
			$calendars[] = $this->createCalendar($calendar->ics_id, $calendar->label, '');
		}

		return $calendars;
	}
}
