<?php
/**
 * @package   DPCalendar
 * @copyright Copyright (C) 2018 Digital Peak GmbH. <https://www.digital-peak.com>
 * @license   https://www.gnu.org/licenses/gpl-3.0.html GNU/GPL
 */

namespace DigitalPeak\Plugin\SampleData\DPCalendar\Extension;

\defined('_JEXEC') or die();

use DigitalPeak\Component\DPCalendar\Administrator\Helper\DPCalendarHelper;
use Joomla\CMS\Application\ApplicationHelper;
use Joomla\CMS\Application\CMSWebApplicationInterface;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Language\Language;
use Joomla\CMS\Language\LanguageFactoryInterface;
use Joomla\CMS\Language\LanguageHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Factory\MVCFactoryInterface;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Component\Menus\Administrator\Model\ItemModel;
use Joomla\Database\DatabaseAwareTrait;

class DPCalendar extends CMSPlugin
{
	use DatabaseAwareTrait;

	protected $autoloadLanguage = true;

	private static string $lorem = '<p>Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo consequat. Duis aute irure dolor in reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla pariatur. Excepteur sint occaecat cupidatat non proident, sunt in culpa qui officia deserunt mollit anim id est laborum.</p>';

	private static array $europeanDateFormatLanguages = ['de-DE', 'fr-FR'];

	private array $languageCache = [];

	private MVCFactoryInterface $factory;

	private LanguageFactoryInterface $languageFactory;

	public function onSampledataGetOverview(): \stdClass
	{
		$data              = new \stdClass();
		$data->name        = $this->_name;
		$data->title       = Text::_('PLG_SAMPLEDATA_DPCALENDAR_OVERVIEW_TITLE');
		$data->description = Text::_('PLG_SAMPLEDATA_DPCALENDAR_OVERVIEW_DESC');
		$data->icon        = 'calendar';
		$data->steps       = 9;

		return $data;
	}

	public function onAjaxSampledataApplyStep1(): array
	{
		$app = $this->getApplication();
		if (!$app instanceof CMSWebApplicationInterface) {
			return [];
		}

		if ($app->getInput()->get('type') != $this->_name) {
			return [];
		}

		try {
			$this->setUp();

			if ($this->params->get('erase')) {
				$this->clearTable('locations');
				$this->clearTable('events');
				$this->clearTable('events_location');
				$this->clearTable('events_hosts');
				$this->clearTable('bookings');
				$this->clearTable('tickets');
				$this->clearTable('taxrates');
				$this->clearTable('coupons');
				$this->clearTable('caldav_calendars');
				$this->clearTable('caldav_calendarinstances');

				$this->getDatabase()->setQuery("delete from #__dpcalendar_extcalendars where plugin = 'ical' or plugin = 'google'");
				$this->getDatabase()->execute();

				$this->getDatabase()->setQuery("delete from #__categories where extension = 'com_dpcalendar'");
				$this->getDatabase()->execute();

				$this->getDatabase()->setQuery("delete from #__fields_values where field_id in (select id from #__fields where context like 'com_dpcalendar.event' or context like 'com_dpcalendar.categories')");
				$this->getDatabase()->execute();
				$this->getDatabase()->setQuery("delete from #__fields where context like 'com_dpcalendar.event' or context like 'com_dpcalendar.categories'");
				$this->getDatabase()->execute();

				$this->getDatabase()->setQuery("delete from #__menu where link like '%com_dpcalendar%' and client_id = 0");
				$this->getDatabase()->execute();

				$this->getDatabase()->setQuery("delete from #__menu_types where menutype like 'dpcalendar-%' and client_id = 0");
				$this->getDatabase()->execute();

				$this->getDatabase()->setQuery("delete from #__modules where module like 'mod_dpcalendar%' or title like '%DPCalendar%'");
				$this->getDatabase()->execute();
			}

			$app->setUserState('sampledata.dpcalendar.events', []);
			$app->setUserState('sampledata.dpcalendar.calendars', []);
			$app->setUserState('sampledata.dpcalendar.locations', []);

			$ids   = [];
			$ids[] = $this->createLocation([
				'title'     => 'Los  Angeles',
				'country'   => 'US',
				'province'  => 'California',
				'city'      => 'Los Angeles',
				'zip'       => '90012',
				'street'    => 'North Alameda Street',
				'number'    => '301-399',
				'latitude'  => '34.05084950',
				'longitude' => '-118.23809670'
			]);
			$ids[] = $this->createLocation([
				'title'     => 'New York',
				'country'   => 'US',
				'province'  => 'New York',
				'city'      => 'New York',
				'zip'       => '10007',
				'street'    => 'Broadway',
				'number'    => '230',
				'latitude'  => '40.71276550',
				'longitude' => '-74.00599370'
			]);
			$ids[] = $this->createLocation([
				'title'     => 'Washington',
				'country'   => 'US',
				'province'  => 'District of Columbia',
				'city'      => 'Washington',
				'zip'       => '20502',
				'street'    => 'Ellipse Road Northwest',
				'latitude'  => '38.89519300',
				'longitude' => '-77.03662770'
			]);
			$ids[] = $this->createLocation([
				'title'     => 'Chicago',
				'country'   => 'US',
				'province'  => 'Illinois',
				'city'      => 'Chicago',
				'zip'       => '60604',
				'street'    => 'West Jackson Boulevard',
				'number'    => '53',
				'latitude'  => '41.87797870',
				'longitude' => '-87.62956640'
			]);
			$ids[] = $this->createLocation([
				'title'     => 'Indianapolis',
				'country'   => 'US',
				'province'  => 'Indiana',
				'city'      => 'Indianapolis',
				'zip'       => '46204',
				'street'    => 'Monument Circle',
				'number'    => '1861',
				'latitude'  => '39.76849080',
				'longitude' => '-86.15767950'
			]);
			$ids[] = $this->createLocation([
				'title'     => 'Rocky Mountain',
				'country'   => 'US',
				'province'  => 'Oklahoma',
				'city'      => 'Rocky Mountain',
				'latitude'  => '35.80536630',
				'longitude' => '-94.76744860'
			]);
			$ids[] = $this->createLocation([
				'title'     => 'Event Venue',
				'country'   => 'US',
				'province'  => 'New York',
				'city'      => 'New York',
				'zip'       => '10123',
				'street'    => 'Les Pilates',
				'number'    => '571',
				'rooms'     => '{"rooms0":{"id":"1","title":"Thor","description":""},"rooms2":{"id":"2","title":"Wolverine","description":""},"rooms3":{"id":"3","title":"Hulk","description":""}}',
				'latitude'  => '40.75200000',
				'longitude' => '-73.99300000',
			]);

			$ids[] = $this->createLocation([
				'title'     => 'London',
				'country'   => 'GB',
				'city'      => 'London',
				'street'    => 'South Carriage Drive',
				'latitude'  => '51.50288180',
				'longitude' => '-0.15714460'
			]);
			$ids[] = $this->createLocation([
				'title'     => 'Berlin',
				'country'   => 'DE',
				'province'  => 'Berlin',
				'city'      => 'Berlin',
				'zip'       => '10178',
				'street'    => 'Tunnel Alexanderplatz',
				'number'    => '9',
				'latitude'  => '52.52248280',
				'longitude' => '13.41158260'
			]);
			$ids[] = $this->createLocation([
				'title'     => 'Mailand',
				'country'   => 'IT',
				'province'  => 'Lombardia',
				'city'      => 'Milano',
				'zip'       => '20122',
				'street'    => 'Piazza del Duomo',
				'number'    => 16,
				'latitude'  => '45.46369490',
				'longitude' => '9.19220070'
			]);
			$ids[] = $this->createLocation([
				'title'     => 'Zurich, Switzerland',
				'country'   => 'CH',
				'province'  => 'Zürich',
				'city'      => 'Zürich',
				'zip'       => '8001',
				'street'    => 'Bahnhofstrasse',
				'number'    => '9-11',
				'latitude'  => '47.36837190',
				'longitude' => '8.53981550'
			]);
			$ids[] = $this->createLocation([
				'title'     => 'Bern',
				'country'   => 'CH',
				'province'  => 'Bern',
				'city'      => 'Bern',
				'zip'       => '3011',
				'street'    => 'Münsterplatz',
				'number'    => '1',
				'latitude'  => '46.94720200',
				'longitude' => '7.45121710'
			]);
			$ids[] = $this->createLocation([
				'title'     => 'Genf',
				'country'   => 'CH',
				'province'  => 'Genève',
				'city'      => 'Genève',
				'zip'       => '1204',
				'street'    => 'Rue Guillaume-Farel',
				'number'    => '8',
				'latitude'  => '46.20078940',
				'longitude' => '6.14889350'
			]);
			$ids[] = $this->createLocation([
				'title'     => 'Lisbon, Portugal',
				'country'   => 'PT',
				'province'  => 'Lisbon',
				'city'      => 'Lisbon',
				'latitude'  => '38.72225240',
				'longitude' => '-9.13933660'
			]);

			$app->setUserState('sampledata.dpcalendar.locations', $ids);

			$response            = [];
			$response['success'] = true;
			$response['message'] = Text::_('PLG_SAMPLEDATA_DPCALENDAR_STEP1_SUCCESS');
		} catch (\Exception $exception) {
			$response            = [];
			$response['success'] = false;
			$response['message'] = Text::sprintf('PLG_SAMPLEDATA_DPCALENDAR_STEP_FAILED', 1, $exception->getMessage());
		}

		return $response;
	}

	public function onAjaxSampledataApplyStep2(): array
	{
		$app = $this->getApplication();
		if (!$app instanceof CMSWebApplicationInterface) {
			return [];
		}

		if ($app->getInput()->get('type') != $this->_name) {
			return [];
		}

		try {
			$this->setUp();

			$fields    = [];
			$fieldData = $this->createCustomField([
				'title'   => 'PLG_SAMPLEDATA_DPCALENDAR_FIELD_4_TITLE',
				'label'   => 'PLG_SAMPLEDATA_DPCALENDAR_FIELD_4_TITLE',
				'type'    => 'text',
				'context' => 'com_dpcalendar.categories'
			]);

			foreach ($fieldData as $code => $name) {
				$fields[$code][] = ['name' => $name, 'value' => 'PLG_SAMPLEDATA_DPCALENDAR_FIELD_4_VALUE'];
			}

			$this->createCalendar([
				'title'       => 'PLG_SAMPLEDATA_DPCALENDAR_CALENDAR_1_TITLE',
				'description' => self::$lorem,
				'fields'      => $fields
			]);
			$this->createCalendar(['title' => 'PLG_SAMPLEDATA_DPCALENDAR_CALENDAR_2_TITLE', 'params' => ['color' => '255436']]);
			$this->createCalendar(['title' => 'PLG_SAMPLEDATA_DPCALENDAR_CALENDAR_3_TITLE', 'params' => ['color' => '9c1f06']]);

			if (!DPCalendarHelper::isFree()) {
				$this->getDatabase()->setQuery("update #__extensions set enabled = 1 where name = 'plg_dpcalendar_ical' or name = 'plg_dpcalendar_google'");
				$this->getDatabase()->execute();

				$this->createExternalCalendar([
					'title'  => 'PLG_SAMPLEDATA_DPCALENDAR_CALENDAR_ICAL_TITLE',
					'plugin' => 'ical',
					'color'  => 'c61111',
					'params' => ['uri' => 'plugins/dpcalendar/ical/examples/calendar.ics']
				]);

				$configFile = JPATH_ROOT . '/DPCalendarGoogleConfig.json';
				if (file_exists($configFile) && PluginHelper::getPlugin('dpcalendar', 'google')) {
					$config = json_decode(file_get_contents($configFile) ?: '');
					foreach ($config->calendars as $cal) {
						$this->createExternalCalendar([
							'title'  => 'PLG_SAMPLEDATA_DPCALENDAR_CALENDAR_GOOGLE_TITLE',
							'plugin' => 'google',
							'color'  => '1541ef',
							'params' => [
								'calendarId'    => $cal->calendarId,
								'refreshToken'  => $cal->refreshToken,
								'client-id'     => $cal->clientId,
								'client-secret' => $cal->clientSecret
							]
						]);
					}
				}

				$this->createPrivateCalendar([
					'displayname'   => 'PLG_SAMPLEDATA_DPCALENDAR_CALENDAR_PRIVATE_TITLE',
					'calendarcolor' => '1dc611'
				]);
			}

			$response            = [];
			$response['success'] = true;
			$response['message'] = Text::_('PLG_SAMPLEDATA_DPCALENDAR_STEP2_SUCCESS');
		} catch (\Exception $exception) {
			$response            = [];
			$response['success'] = false;
			$response['message'] = Text::sprintf('PLG_SAMPLEDATA_DPCALENDAR_STEP_FAILED', 2, $exception->getMessage());
		}

		return $response;
	}

	public function onAjaxSampledataApplyStep3(): array
	{
		$app = $this->getApplication();
		if (!$app instanceof CMSWebApplicationInterface) {
			return [];
		}

		if ($app->getInput()->get('type') != $this->_name) {
			return [];
		}

		try {
			$this->setUp();

			$locationIds = $app->getUserState('sampledata.dpcalendar.locations');

			// Weekly
			$this->createEvent([
				'catid'          => 0,
				'title'          => 'PLG_SAMPLEDATA_DPCALENDAR_EVENT_1_TITLE',
				'rrule'          => 'FREQ=WEEKLY;BYDAY=MO',
				'color'          => 'CC2B40',
				'url'            => 'http://www.digital-peak.com',
				'description'    => 'PLG_SAMPLEDATA_DPCALENDAR_EVENT_1_DESC',
				'access_content' => 3
			]);

			// Two days
			$start = DPCalendarHelper::getDate();
			if (!DPCalendarHelper::isFree()) {
				$start->modify('-1 month');
			}
			$start->setTime(13, 0, 0);

			$end = clone $start;
			$end->modify('+2 hours');
			$end->modify('+1 day');
			$this->createEvent([
				'catid'        => 0,
				'title'        => 'PLG_SAMPLEDATA_DPCALENDAR_EVENT_2_TITLE',
				'rrule'        => 'FREQ=WEEKLY;BYDAY=WE',
				'start_date'   => $start->toSql(),
				'end_date'     => $end->toSql(),
				'description'  => 'PLG_SAMPLEDATA_DPCALENDAR_EVENT_2_DESC',
				'location_ids' => $locationIds[0]
			]);

			// Full day
			$this->createEvent([
				'catid'       => 0,
				'title'       => 'PLG_SAMPLEDATA_DPCALENDAR_EVENT_3_TITLE',
				'rrule'       => 'FREQ=WEEKLY;BYDAY=TH',
				'all_day'     => '1',
				'color'       => 'B310CC',
				'description' => 'PLG_SAMPLEDATA_DPCALENDAR_EVENT_3_DESC'
			]);

			// Multi day
			$this->createEvent([
				'catid'        => 0,
				'title'        => 'PLG_SAMPLEDATA_DPCALENDAR_EVENT_4_TITLE',
				'rrule'        => 'FREQ=WEEKLY;BYDAY=SA',
				'color'        => 'FF9442',
				'description'  => 'PLG_SAMPLEDATA_DPCALENDAR_EVENT_4_DESC',
				'location_ids' => [$locationIds[3], $locationIds[4]]
			]);

			$response            = [];
			$response['success'] = true;
			$response['message'] = Text::_('PLG_SAMPLEDATA_DPCALENDAR_STEP3_SUCCESS');
		} catch (\Exception $exception) {
			$response            = [];
			$response['success'] = false;
			$response['message'] = Text::sprintf('PLG_SAMPLEDATA_DPCALENDAR_STEP_FAILED', 3, $exception->getMessage());
		}

		return $response;
	}

	public function onAjaxSampledataApplyStep4(): array
	{
		$app = $this->getApplication();
		if (!$app instanceof CMSWebApplicationInterface) {
			return [];
		}

		if ($app->getInput()->get('type') != $this->_name) {
			return [];
		}

		try {
			$this->setUp();

			$locationIds = $app->getUserState('sampledata.dpcalendar.locations');

			$this->createEvent([
				'catid'                => 1,
				'title'                => 'PLG_SAMPLEDATA_DPCALENDAR_EVENT_6_TITLE',
				'rrule'                => 'FREQ=WEEKLY;BYDAY=SA',
				'images'               => '{"image_intro":"media\\/plg_sampledata_dpcalendar\\/images\\/festival.jpg","image_intro_alt":"Festival","image_intro_caption":"","image_full":"media\\/plg_sampledata_dpcalendar\\/images\\/festival.jpg","image_full_alt":"Festival","image_full_caption":"","image_intro_width":800,"image_intro_height":480,"image_full_width":800,"image_full_height":480}',
				'color'                => 'ebb31a',
				'capacity'             => '80',
				'capacity_used'        => '80',
				'max_tickets'          => 2,
				'booking_waiting_list' => 1,
				'booking_series'       => 1,
				'booking_closing_date' => '+2 months',
				'prices'               => '{"prices0":{"value":"45.00","label":"","description":"","currency":"EUR"}}',
				'booking_information'  => '<p>Every attendee needs to bring his own clothes.</p>',
				'location_ids'         => $locationIds[4]
			]);
			$this->createEvent([
				'catid'                => 1,
				'title'                => 'PLG_SAMPLEDATA_DPCALENDAR_EVENT_7_TITLE',
				'rrule'                => 'FREQ=WEEKLY;BYDAY=TH',
				'images'               => '{"image_intro":"media\\/plg_sampledata_dpcalendar\\/images\\/hike.jpg","image_intro_alt":"Hike","image_intro_caption":"","image_full":"media\\/plg_sampledata_dpcalendar\\/images\\/hike.jpg","image_full_alt":"Hike","image_full_caption":"","image_intro_width":800,"image_intro_height":457,"image_full_width":800,"image_full_height":457}',
				'color'                => '127a0f',
				'capacity'             => null,
				'max_tickets'          => 100,
				'prices'               => '{"prices0":{"value":"20.00","label":"","description":"","currency":"EUR"}}',
				'booking_series'       => 1,
				'tickets_discount'     => '{"tickets_discount0":{"value":"25","type":"percentage","amount":2,"label":"","description":""}}',
				'booking_closing_date' => DPCalendarHelper::getDate('+' . $this->params->get('until', '6 month'))->format('Y-m-d 235900'),
				'location_ids'         => $locationIds[5]
			]);
			$this->createEvent([
				'catid'        => 1,
				'title'        => 'PLG_SAMPLEDATA_DPCALENDAR_EVENT_8_TITLE',
				'rrule'        => 'FREQ=WEEKLY;BYDAY=SA',
				'images'       => '{"image_intro":"media\\/plg_sampledata_dpcalendar\\/images\\/swimming.jpg","image_intro_alt":"Swimming","image_intro_caption":"","image_full":"media\\/plg_sampledata_dpcalendar\\/images\\/swimming.jpg","image_full_alt":"Swimming","image_full_caption":"","image_intro_width":800,"image_intro_height":535,"image_full_width":800,"image_full_height":535}',
				'schedule'     => '{"schedule0":{"title":"Intro","duration":"10","description":"Welcome the attendees."},"schedule1":{"title":"Warmup","duration":"20","description":"Making yourself ready."},"schedule2":{"title":"Exercise","duration":"60","description":"Training the different styles."},"schedule3":{"title":"Feedback","duration":"10","description":"Discussion round amongst the attendees."}}',
				'capacity'     => null,
				'max_tickets'  => 20,
				'location_ids' => $locationIds[0]
			]);
			$this->createEvent([
				'catid'                => 1,
				'title'                => 'PLG_SAMPLEDATA_DPCALENDAR_EVENT_9_TITLE',
				'rrule'                => 'FREQ=WEEKLY;BYDAY=SU',
				'images'               => '{"image_intro":"media\\/plg_sampledata_dpcalendar\\/images\\/basketball.jpg","image_intro_alt":"Basketball","image_intro_caption":"","image_full":"media\\/plg_sampledata_dpcalendar\\/images\\/basketball.jpg","image_full_alt":"Basketball","image_full_caption":"","image_intro_width":800,"image_intro_height":351,"image_full_width":800,"image_full_height":351}',
				'color'                => 'c42323',
				'capacity'             => '15',
				'max_tickets'          => 5,
				'booking_waiting_list' => 1,
				'prices'               => '{"prices0":{"value":"55","label":"Kids","description":"Age: 1 - 5 years","currency":"EUR"},"prices1":{"value":"68","label":"Student","description":"Needs an ID","currency":"EUR"},"prices2":{"value":"89","label":"Adults","description":"Age: Older than 6 years","currency":"EUR"}}',
				'earlybird_discount'   => '{"earlybird_discount0":{"value":"20","type":"percentage","date":"-2 days","label":"Early Bird Discount","description":"Decide early, pay less"}}',
				'booking_options'      => '{"booking_options0":{"value":"15","amount":"1","label":"Lunch box small","description":"A small snack","currency":"EUR"},"booking_options1":{"value":"25","amount":"1","label":"Lunch box big","description":"For the hungry ones","currency":"EUR"}}',
				'location_ids'         => $locationIds[1]
			]);

			$this->createEvent([
				'catid'               => 1,
				'title'               => 'PLG_SAMPLEDATA_DPCALENDAR_EVENT_10_TITLE',
				'rrule'               => 'FREQ=WEEKLY;BYDAY=SA',
				'images'              => '{"image_intro":"media\\/plg_sampledata_dpcalendar\\/images\\/rock-concert.jpg","image_intro_alt":"Rock concert","image_intro_caption":"","image_full":"media\\/plg_sampledata_dpcalendar\\/images\\/rock-concert.jpg","image_full_alt":"Rock concert","image_full_caption":"","image_intro_width":800,"image_intro_height":533,"image_full_width":800,"image_full_height":533}',
				'description'         => 'PLG_SAMPLEDATA_DPCALENDAR_EVENT_10_DESC',
				'color'               => '3c3d3c',
				'capacity'            => '80',
				'max_tickets'         => 1,
				'state'               => 3,
				'prices'              => '{"prices0":{"value":"15.00","label":"","description":"",,"currency":"EUR"}}',
				'booking_information' => '',
				'location_ids'        => $locationIds[1]
			]);

			$response            = [];
			$response['success'] = true;
			$response['message'] = Text::_('PLG_SAMPLEDATA_DPCALENDAR_STEP4_SUCCESS');
		} catch (\Exception $exception) {
			$response            = [];
			$response['success'] = false;
			$response['message'] = Text::sprintf('PLG_SAMPLEDATA_DPCALENDAR_STEP_FAILED', 4, $exception->getMessage());
		}

		return $response;
	}

	public function onAjaxSampledataApplyStep5(): array
	{
		$app = $this->getApplication();
		if (!$app instanceof CMSWebApplicationInterface) {
			return [];
		}

		if ($app->getInput()->get('type') != $this->_name) {
			return [];
		}

		try {
			$this->setUp();

			$locationIds = $app->getUserState('sampledata.dpcalendar.locations');

			$start = DPCalendarHelper::getDate();
			$start->setTime(8, 0, 0);
			$end = clone $start;
			$end->modify('+4 hours');

			$this->createEvent([
				'catid'        => 2,
				'title'        => 'PLG_SAMPLEDATA_DPCALENDAR_EVENT_11_TITLE',
				'rrule'        => 'FREQ=DAILY',
				'start_date'   => $start->toSql(),
				'end_date'     => $end->toSql(),
				'color'        => '3366CC',
				'description'  => 'PLG_SAMPLEDATA_DPCALENDAR_EVENT_11_DESC',
				'rooms'        => $locationIds[6] . '-1',
				'location_ids' => $locationIds[6]
			]);
			$this->createEvent([
				'catid'        => 2,
				'title'        => 'PLG_SAMPLEDATA_DPCALENDAR_EVENT_12_TITLE',
				'rrule'        => 'FREQ=DAILY',
				'start_date'   => $start->toSql(),
				'end_date'     => $end->toSql(),
				'color'        => 'FF4557',
				'description'  => 'PLG_SAMPLEDATA_DPCALENDAR_EVENT_12_DESC',
				'rooms'        => $locationIds[6] . '-2',
				'location_ids' => $locationIds[6]
			]);
			$this->createEvent([
				'catid'        => 2,
				'title'        => 'PLG_SAMPLEDATA_DPCALENDAR_EVENT_13_TITLE',
				'rrule'        => 'FREQ=DAILY',
				'start_date'   => $start->toSql(),
				'end_date'     => $end->toSql(),
				'color'        => '056625',
				'description'  => 'PLG_SAMPLEDATA_DPCALENDAR_EVENT_13_DESC',
				'rooms'        => $locationIds[6] . '-3',
				'location_ids' => $locationIds[6]
			]);

			$response            = [];
			$response['success'] = true;
			$response['message'] = Text::_('PLG_SAMPLEDATA_DPCALENDAR_STEP5_SUCCESS');
		} catch (\Exception $exception) {
			$response            = [];
			$response['success'] = false;
			$response['message'] = Text::sprintf('PLG_SAMPLEDATA_DPCALENDAR_STEP_FAILED', 5, $exception->getMessage());
		}

		return $response;
	}

	public function onAjaxSampledataApplyStep6(): array
	{
		$app = $this->getApplication();
		if (!$app instanceof CMSWebApplicationInterface) {
			return [];
		}

		if ($app->getInput()->get('type') != $this->_name) {
			return [];
		}

		try {
			$this->setUp();

			$fields    = [];
			$fieldData = $this->createCustomField([
				'title'   => 'PLG_SAMPLEDATA_DPCALENDAR_FIELD_1_TITLE',
				'label'   => 'PLG_SAMPLEDATA_DPCALENDAR_FIELD_1_TITLE',
				'type'    => 'text',
				'context' => 'com_dpcalendar.event'
			]);

			foreach ($fieldData as $code => $name) {
				$fields[$code][] = ['name' => $name, 'value' => 'PLG_SAMPLEDATA_DPCALENDAR_FIELD_1_VALUE'];
			}

			$fieldData = $this->createCustomField([
				'title'   => 'PLG_SAMPLEDATA_DPCALENDAR_FIELD_2_TITLE',
				'label'   => 'PLG_SAMPLEDATA_DPCALENDAR_FIELD_2_TITLE',
				'type'    => 'media',
				'context' => 'com_dpcalendar.event'
			]);

			foreach ($fieldData as $code => $name) {
				$fields[$code][] = ['name' => $name, 'value' => 'images/powered_by.png'];
			}

			$fieldData = $this->createCustomField([
				'title'   => 'PLG_SAMPLEDATA_DPCALENDAR_FIELD_3_TITLE',
				'label'   => 'PLG_SAMPLEDATA_DPCALENDAR_FIELD_3_TITLE',
				'type'    => 'dpcalendar',
				'context' => 'com_dpcalendar.event'
			]);

			foreach ($fieldData as $code => $name) {
				$fields[$code][] = ['name' => $name, 'value' => $app->getUserState('sampledata.dpcalendar.calendars', [])[1][$code]];
			}

			// Custom fields
			$this->createEvent([
				'catid'       => 0,
				'title'       => 'PLG_SAMPLEDATA_DPCALENDAR_EVENT_5_TITLE',
				'rrule'       => 'FREQ=WEEKLY;BYDAY=SU',
				'description' => 'PLG_SAMPLEDATA_DPCALENDAR_EVENT_5_DESC',
				'fields'      => $fields
			]);

			$response            = [];
			$response['success'] = true;
			$response['message'] = Text::_('PLG_SAMPLEDATA_DPCALENDAR_STEP6_SUCCESS');
		} catch (\Exception $exception) {
			$response            = [];
			$response['success'] = false;
			$response['message'] = Text::sprintf('PLG_SAMPLEDATA_DPCALENDAR_STEP_FAILED', 6, $exception->getMessage());
		}

		return $response;
	}

	public function onAjaxSampledataApplyStep7(): array
	{
		$app = $this->getApplication();
		if (!$app instanceof CMSWebApplicationInterface) {
			return [];
		}

		if ($app->getInput()->get('type') != $this->_name || DPCalendarHelper::isFree()) {
			return ['success' => true, 'message' => Text::_('PLG_SAMPLEDATA_DPCALENDAR_STEP7_SUCCESS')];
		}

		try {
			$this->setUp();

			// Create tax rate
			$model = $this->factory->createModel('Country', 'Administrator', ['ignore_request' => true]);
			if ($country = $model->getItem(['short_code' => 'US'])) {
				$country = $country->id;
			}
			$this->factory->createModel('Taxrate', 'Administrator', ['ignore_request' => true])->save([
				'title'     => 'VAT',
				'rate'      => 10,
				'state'     => 1,
				'countries' => '{"countries0":{"country":"' . ($country ?: 0) . '"}}'
			]);

			// Create coupon
			$calendarIds = [];
			foreach ($app->getUserState('sampledata.dpcalendar.calendars', []) as $calendars) {
				foreach ($calendars as $calendarId) {
					if (!is_numeric($calendarId)) {
						continue;
					}
					$calendarIds[] = $calendarId;
				}
			}
			$couponModel = $this->factory->createModel('Coupon', 'Administrator', ['ignore_request' => true]);
			$couponModel->save([
				'title'     => 'Coupon',
				'code'      => 'DEMO',
				'value'     => 20,
				'type'      => 'percentage',
				'calendars' => implode(',', $calendarIds),
				'state'     => 1
			]);

			// Festival
			$this->createBooking([
				'first_name'       => 'Chuck',
				'name'             => 'Norris',
				'email'            => 'chuck@example.com',
				'country'          => 'US',
				'city'             => 'Texas',
				'latitude'         => 31.81603810,
				'longitude'        => -99.51209860,
				'event_id'         => [4 => ['tickets' => [0 => 1]]],
				'state'            => 8,
				'coupon_id'        => 'DEMO',
				'payment_provider' => 'manual-1'
			]);

			// Hike
			$this->createBooking([
				'first_name'       => 'Arnold',
				'name'             => 'Schwarzenegger',
				'email'            => 'arnold@example.com',
				'country'          => 'US',
				'city'             => 'Texas',
				'latitude'         => 31.81603810,
				'longitude'        => -99.51209860,
				'event_id'         => [5 => ['tickets' => [0 => 2]]],
				'state'            => 1,
				'payment_provider' => 'manual-1',
				'invoice'          => 1
			]);

			// Swimming
			$this->createBooking([
				'first_name' => 'Jean-Claude',
				'name'       => 'van Damme',
				'email'      => 'jean@example.com',
				'country'    => 'BE',
				'city'       => 'Sint-Agatha-Berchem',
				'latitude'   => 50.86492310,
				'longitude'  => 4.29467340,
				'event_id'   => [6 => ['tickets' => [0 => 1]]],
				'state'      => 1
			]);

			foreach ($this->languageCache as $language) {
				$table = $this->factory->createTable('Ticket', 'Administrator');
				$table->load(['email' => 'jean@example.com']);
				$table->state = 9;
				$table->store();
			}

			// Basketball
			$this->createBooking([
				'first_name'       => 'Bruce',
				'name'             => 'Lee',
				'email'            => 'lee@example.com',
				'country'          => 'US',
				'city'             => 'New York',
				'latitude'         => 40.71272810,
				'longitude'        => -74.00601520,
				'event_id'         => [7 => ['tickets' => [0 => 2, 1 => 0, 2 => 1]]],
				'state'            => 2,
				'payment_provider' => 'manual-1'
			]);

			$response            = [];
			$response['success'] = true;
			$response['message'] = Text::_('PLG_SAMPLEDATA_DPCALENDAR_STEP7_SUCCESS');
		} catch (\Exception $exception) {
			$response            = [];
			$response['success'] = false;
			$response['message'] = Text::sprintf('PLG_SAMPLEDATA_DPCALENDAR_STEP_FAILED', 7, $exception->getMessage());
		}

		return $response;
	}

	public function onAjaxSampledataApplyStep8(): array
	{
		$app = $this->getApplication();
		if (!$app instanceof CMSWebApplicationInterface) {
			return [];
		}

		if ($app->getInput()->get('type') != $this->_name) {
			return [];
		}

		if (!ComponentHelper::isEnabled('com_menus')) {
			return ['success' => true, 'message' => Text::sprintf('PLG_SAMPLEDATA_DPCALENDAR_STEP_SKIPPED', 8, 'com_menus')];
		}

		try {
			$this->setUp();

			if (\count($this->languageCache) > 1) {
				foreach (array_keys($this->languageCache) as $code) {
					$menuTable = $app->bootComponent('menus')->getMVCFactory()->createTable('MenuType', 'Administrator');

					$menuTable->bind([
						'menutype'    => 'dpcalendar-' . $code,
						'title'       => 'DPCalendar ' . $code,
						'description' => ''
					]);

					$menuTable->store();
				}
			}

			$locationIds = $app->getUserState('sampledata.dpcalendar.locations');

			$this->createMenuItem([
				'title'  => 'PLG_SAMPLEDATA_DPCALENDAR_MENU_ITEM_1_TITLE',
				'link'   => 'view=calendar',
				'home'   => \count($this->languageCache) > 1 ? 1 : 0,
				'params' => ['ids' => [0]]
			]);
			if (!DPCalendarHelper::isFree()) {
				$this->createMenuItem([
					'title'  => 'PLG_SAMPLEDATA_DPCALENDAR_MENU_ITEM_2_TITLE',
					'link'   => 'view=calendar&layout=timeline',
					'params' => ['ids' => [0, 1]]
				]);
			}
			$this->createMenuItem([
				'title' => 'PLG_SAMPLEDATA_DPCALENDAR_MENU_ITEM_3_TITLE',
				'link'  => 'view=event&id=',
				'id'    => 4
			]);

			if (!DPCalendarHelper::isFree()) {
				$this->createMenuItem([
					'title'  => 'PLG_SAMPLEDATA_DPCALENDAR_MENU_ITEM_4_TITLE',
					'link'   => 'view=list',
					'params' => ['ids' => [1]]
				]);
				$this->createMenuItem([
					'title'  => 'PLG_SAMPLEDATA_DPCALENDAR_MENU_ITEM_5_TITLE',
					'link'   => 'view=list&layout=blog',
					'params' => ['ids' => [1]]
				]);
				$this->createMenuItem([
					'title'  => 'PLG_SAMPLEDATA_DPCALENDAR_MENU_ITEM_6_TITLE',
					'link'   => 'view=list&layout=timeline',
					'params' => ['ids' => [1]]
				]);
				$this->createMenuItem([
					'title'  => 'PLG_SAMPLEDATA_DPCALENDAR_MENU_ITEM_7_TITLE',
					'link'   => 'view=map',
					'params' => ['ids' => [0], 'map_filter_radius' => '-1']
				]);
			}
			$this->createMenuItem([
				'title'  => 'PLG_SAMPLEDATA_DPCALENDAR_MENU_ITEM_8_TITLE',
				'link'   => 'view=locations',
				'params' => ['ids' => [$locationIds[0], $locationIds[1]]]
			]);
			$this->createMenuItem([
				'title' => 'PLG_SAMPLEDATA_DPCALENDAR_MENU_ITEM_9_TITLE',
				'link'  => 'view=location&id=' . $locationIds[6]
			]);

			if (!DPCalendarHelper::isFree()) {
				$this->createMenuItem([
					'title'  => 'PLG_SAMPLEDATA_DPCALENDAR_MENU_ITEM_10_TITLE',
					'link'   => 'view=calendar',
					'params' => [
						'ids'                       => [2],
						'default_view'              => 'day',
						'header_show_month'         => 0,
						'header_show_week'          => 0,
						'header_show_day'           => 0,
						'header_show_list'          => 0,
						'min_time'                  => '6:00',
						'max_time'                  => '20:00',
						'business_hours_start'      => '7:00',
						'business_hours_end'        => '17:00',
						'business_hours_days'       => [1, 2, 3, 4, 5, 6, 0],
						'calendar_filter_locations' => [$locationIds[6]],
						'calendar_resource_views'   => ['day']
					]
				]);
				$this->createMenuItem([
					'title' => 'PLG_SAMPLEDATA_DPCALENDAR_MENU_ITEM_11_TITLE',
					'link'  => 'view=bookings'
				]);
				$this->createMenuItem([
					'title' => 'PLG_SAMPLEDATA_DPCALENDAR_MENU_ITEM_12_TITLE',
					'link'  => 'view=tickets'
				]);
				$this->createMenuItem([
					'title'  => 'PLG_SAMPLEDATA_DPCALENDAR_MENU_ITEM_13_TITLE',
					'link'   => 'view=calendar',
					'params' => [
						'ids'                  => [3, 4],
						'textbefore'           => 'PLG_SAMPLEDATA_DPCALENDAR_MENU_ITEM_13_TEXT',
						'calendar_filter_form' => 2,
						'map_zoom'             => 3
					]
				]);
				$this->createMenuItem([
					'title' => 'PLG_SAMPLEDATA_DPCALENDAR_MENU_ITEM_14_TITLE',
					'link'  => 'view=profile'
				]);
			}

			$response            = [];
			$response['success'] = true;
			$response['message'] = Text::_('PLG_SAMPLEDATA_DPCALENDAR_STEP8_SUCCESS');
		} catch (\Exception $exception) {
			$response            = [];
			$response['success'] = false;
			$response['message'] = Text::sprintf('PLG_SAMPLEDATA_DPCALENDAR_STEP_FAILED', 8, $exception->getMessage());
		}

		return $response;
	}

	public function onAjaxSampledataApplyStep9(): array
	{
		$app = $this->getApplication();
		if (!$app instanceof CMSWebApplicationInterface) {
			return [];
		}

		if ($app->getInput()->get('type') != $this->_name) {
			return [];
		}

		if (!ComponentHelper::isEnabled('com_modules')) {
			return ['success' => true, 'message' => Text::sprintf('PLG_SAMPLEDATA_DPCALENDAR_STEP_SKIPPED', 9, 'com_modules')];
		}

		try {
			$this->setUp();

			if (\count($this->languageCache) > 1) {
				// Menu module
				$this->createModule([
					'title'    => 'DPCalendar',
					'ordering' => 1,
					'module'   => 'mod_menu'
				]);
			}

			$this->createModule([
				'title'    => 'PLG_SAMPLEDATA_DPCALENDAR_MODULE_1_TITLE',
				'ordering' => 2,
				'module'   => 'mod_dpcalendar_counter',
				'params'   => ['ids' => [0], 'description_length' => '0']
			]);
			$this->createModule([
				'title'    => 'PLG_SAMPLEDATA_DPCALENDAR_MODULE_2_TITLE',
				'ordering' => 3,
				'module'   => 'mod_dpcalendar_upcoming',
				'params'   => ['ids' => [1], 'description_length' => '0']
			]);
			$this->createModule([
				'title'    => 'PLG_SAMPLEDATA_DPCALENDAR_MODULE_3_TITLE',
				'ordering' => 4,
				'module'   => 'mod_dpcalendar_map',
				'params'   => ['ids' => [1], 'radius' => '-1']
			]);
			$this->createModule([
				'title'    => 'PLG_SAMPLEDATA_DPCALENDAR_MODULE_4_TITLE',
				'ordering' => 5,
				'module'   => 'mod_dpcalendar_mini',
				'params'   => [
					'ids'               => [0],
					'header_show_month' => 0,
					'header_show_week'  => 0,
					'header_show_day'   => 0,
					'header_show_list'  => 0,
				]
			]);

			$response            = [];
			$response['success'] = true;
			$response['message'] = Text::_('PLG_SAMPLEDATA_DPCALENDAR_STEP9_SUCCESS');
		} catch (\Exception $exception) {
			$response            = [];
			$response['success'] = false;
			$response['message'] = Text::sprintf('PLG_SAMPLEDATA_DPCALENDAR_STEP_FAILED', 9, $exception->getMessage());
		}

		return $response;
	}

	public function setLanguageFactory(LanguageFactoryInterface $factory): void
	{
		$this->languageFactory = $factory;
	}

	/**
	 * @return string[]
	 */
	private function createCustomField(array $originalData): array
	{
		$app = $this->getApplication();
		if (!$app instanceof CMSWebApplicationInterface) {
			return [];
		}

		$newIds = [];
		foreach ($this->languageCache as $code => $language) {
			$data                  = $originalData;
			$data['id']            = 0;
			$data['name']          = strtolower('field-' . $code . '-' . preg_replace('/[^0-9,.]/', '', (string)$data['title']));
			$data['title']         = $language->_($data['title']);
			$data['label']         = $language->_($data['label']);
			$data['language']      = \count($this->languageCache) > 1 ? $code : '*';
			$data['description']   = '';
			$data['params']        = '{}';
			$data['state']         = 1;
			$data['default_value'] = '';
			$data['access']        = (int)$app->get('access', 1);

			$model = $app->bootComponent('fields')->getMVCFactory()->createModel('Field', 'Administrator');
			if (!$model->save($data)) {
				$app->getLanguage()->load('com_fields');
				throw new \Exception(Text::_($model->getError()));
			}
			$newIds[$code] = $data['name'];
		}

		return $newIds;
	}

	private function createModule(array $originalData): void
	{
		$app = $this->getApplication();
		if (!$app instanceof CMSWebApplicationInterface) {
			return;
		}

		$calendarIds = $app->getUserState('sampledata.dpcalendar.calendars', []);
		foreach ($this->languageCache as $code => $language) {
			$data = $originalData;
			if (!empty($data['params']) && !empty($data['params']['ids'])) {
				foreach ($data['params']['ids'] as $index => $id) {
					$data['params']['ids'][$index] = $calendarIds[$id][$code];
				}
			}

			if (\in_array($code, self::$europeanDateFormatLanguages, true)) {
				switch ($data['module']) {
					case 'mod_dpcalendar_map':
					case 'mod_dpcalendar_upcoming':
						$data['params']['date_format'] = 'd.m.Y';
						$data['params']['time_format'] = 'H:i';
						break;
					case 'mod_dpcalendar_upcoming':
						$data['params']['timeformat_month'] = 'H:i';
						$data['params']['timeformat_week']  = 'H:i';
						$data['params']['timeformat_day']   = 'H:i';
						$data['params']['timeformat_list']  = 'H:i';
						break;
				}
			}

			if ($data['module'] == 'mod_menu') {
				$data['params'] = ['menutype' => 'dpcalendar-' . $code];
			}

			$data['id']         = 0;
			$data['title']      = $language->_($data['title']);
			$data['asset_id']   = 0;
			$data['language']   = \count($this->languageCache) > 1 ? $code : '*';
			$data['content']    = '';
			$data['note']       = '';
			$data['published']  = $this->params->get('publish_modules', 0);
			$data['assignment'] = 0;
			$data['access']     = (int)$app->get('access', 1);
			$data['client_id']  = 0;
			$data['position']   = 'sidebar-right';

			$model = $app->bootComponent('modules')->getMVCFactory()->createModel('Module', 'Administrator');
			if (!$model->save($data)) {
				$app->getLanguage()->load('com_modules');
				throw new \Exception(Text::_($model->getError()));
			}
		}
	}

	private function createMenuItem(array $originalData): void
	{
		$app = $this->getApplication();
		if (!$app instanceof CMSWebApplicationInterface) {
			return;
		}

		$this->getDatabase()->setQuery("select extension_id from #__extensions where name ='com_dpcalendar' and type = 'component'");
		$componentId = $this->getDatabase()->loadResult();
		$calendarIds = $app->getUserState('sampledata.dpcalendar.calendars', []);
		$eventIds    = $app->getUserState('sampledata.dpcalendar.events', []);

		foreach ($this->languageCache as $code => $language) {
			$data = $originalData;
			if (!empty($data['params']) && !empty($data['params']['ids']) && $originalData['link'] != 'view=locations') {
				foreach ($data['params']['ids'] as $index => $id) {
					$data['params']['ids'][$index] = $calendarIds[$id][$code];
				}
			}
			if (!empty($data['params']) && !empty($data['params']['textbefore'])) {
				$data['params']['textbefore'] = $language->_($data['params']['textbefore']);
			}

			if (\in_array($code, self::$europeanDateFormatLanguages, true)) {
				switch ($data['link']) {
					case 'view=calendar':
						$data['params']['timeformat_month'] = 'H:i';
						$data['params']['timeformat_week']  = 'H:i';
						$data['params']['timeformat_day']   = 'H:i';
						$data['params']['timeformat_list']  = 'H:i';
						// Params for calendar
						// no break
					case 'view=list':
					case 'view=list&layout=blog':
					case 'view=list&layout=timeline':
						$data['params']['list_title_format'] = 'd.m.Y';
						// Params for list
						// no break
					case 'view=map':
						$data['params']['map_date_format'] = 'H:i';
						// Params for map
						// no break
					case 'view=event&id=':
						$data['params']['event_date_format'] = 'd.m.Y';
						$data['params']['event_time_format'] = 'H:i';
				}
			}

			if (!empty($data['id'])) {
				$data['link'] .= $eventIds[$data['id']][$code];
				unset($data['id']);
			}

			// Set values which are always the same
			$data['id']              = 0;
			$data['title']           = $language->_($data['title']);
			$data['created_user_id'] = $app->getIdentity() !== null ? $app->getIdentity()->id : 0;
			$data['alias']           = ApplicationHelper::stringURLSafe($data['title']);
			$data['link']            = 'index.php?option=com_dpcalendar&' . $data['link'];
			$data['menutype']        = \count($this->languageCache) > 1 ? 'dpcalendar-' . $code : 'mainmenu';
			$data['component_id']    = $componentId;
			$data['language']        = \count($this->languageCache) > 1 ? $code : '*';
			$data['params']          = empty($data['params']) ? '' : $data['params'];

			// Set unicodeslugs if alias is empty
			if (trim(str_replace('-', '', $data['alias'])) !== '' && trim(str_replace('-', '', $data['alias'])) !== '0') {
				$unicode       = $app->set('unicodeslugs', 1);
				$data['alias'] = ApplicationHelper::stringURLSafe($data['title']);
				$app->set('unicodeslugs', $unicode);
			}

			$data['published']         = 1;
			$data['note']              = '';
			$data['img']               = '';
			$data['path']              = '';
			$data['associations']      = [];
			$data['client_id']         = 0;
			$data['level']             = 1;
			$data['browserNav']        = 0;
			$data['access']            = (int)$app->get('access', 1);
			$data['type']              = 'component';
			$data['template_style_id'] = 0;
			$data['parent_id']         = 1;

			/** @var ItemModel $model */
			$model = $app->bootComponent('menus')->getMVCFactory()->createModel('Item', 'Administrator', ['ignore_request' => true]);
			if (!$model->save($data)) {
				// When not fully translated we can have duplicates on the alias
				$data['alias'] = $data['alias'] . '-' . $code;
				if (!$model->save($data)) {
					throw new \Exception($data['title'] . ' => ' . $data['alias'] . ' : ' . $model->getError());
				}
			}
		}
	}

	private function createBooking(array $originalData): void
	{
		$app = $this->getApplication();
		if (!$app instanceof CMSWebApplicationInterface) {
			return;
		}

		if (DPCalendarHelper::isFree()) {
			return;
		}

		$eventIds = $app->getUserState('sampledata.dpcalendar.events', []);

		$originalData['user_id'] = $app->getIdentity() !== null ? $app->getIdentity()->id : 0;
		foreach (array_keys($this->languageCache) as $code) {
			$data = $originalData;
			foreach ($data['event_id'] as $eventId => $tickets) {
				$data['event_id'][$this->getEvent((int)$eventIds[$eventId][$code])->id] = $tickets;
				unset($data['event_id'][$eventId]);
			}

			$model = $this->factory->createModel('Country', 'Administrator', ['ignore_request' => true]);
			if ($country = $model->getItem(['short_code' => $data['country']])) {
				$data['country'] = $country->id;
			}

			// Defaults
			$data['txn_type']     = '';
			$data['txn_currency'] = '';
			$data['payer_id']     = '';
			$data['payer_email']  = '';

			$model = $this->factory->createModel('Booking', 'Administrator', ['ignore_request' => true]);
			if (!$model->save($data)) {
				throw new \Exception(Text::_($model->getError()));
			}

			// Create only one booking
			break;
		}
	}

	private function createEvent(array $originalData): void
	{
		$app = $this->getApplication();
		if (!$app instanceof CMSWebApplicationInterface) {
			return;
		}

		if (\array_key_exists('capacity', $originalData) && DPCalendarHelper::isFree()) {
			unset($originalData['capacity']);
		}

		if (!empty($originalData['location_ids']) && !\is_array($originalData['location_ids'])) {
			$originalData['location_ids'] = [$originalData['location_ids']];
		}
		if (!isset($originalData['state'])) {
			$originalData['state'] = 1;
		}
		$originalData['access'] = (int)$app->get('access', 1);
		if (!isset($originalData['access_content'])) {
			$originalData['access_content'] = $originalData['access'];
		}

		if (!isset($originalData['all_day'])) {
			$originalData['all_day'] = 0;
		}

		if (!\array_key_exists('capacity', $originalData)) {
			$originalData['capacity'] = 0;
		}

		if (DPCalendarHelper::isFree() && !empty($originalData['rrule'])) {
			unset($originalData['rrule']);
		}

		if (!empty($originalData['rrule'])) {
			$originalData['rrule'] .= ';UNTIL=' . DPCalendarHelper::getDate('+' . $this->params->get('until', '6 month'))->format('Ymd\T235900');
		}

		if (empty($originalData['start_date'])) {
			$start = DPCalendarHelper::getDate();
			if (!DPCalendarHelper::isFree()) {
				$start->modify('-1 month');
			} else {
				$start->modify('+' . random_int(1, 7) . ' day');
			}
			$start->modify('+1 hour');
			$start->setTime((int)$start->format('H'), 0, 0);

			$end = clone $start;
			$end->modify('+2 hours');

			$originalData['start_date'] = $start->toSql();
			$originalData['end_date']   = $end->toSql();
		}

		$newIds      = [];
		$calendarIds = $app->getUserState('sampledata.dpcalendar.calendars', []);
		foreach ($this->languageCache as $code => $language) {
			$data                = $originalData;
			$data['catid']       = $calendarIds[$originalData['catid']][$code];
			$data['title']       = $language->_($data['title']);
			$data['host_ids']    = [43, 44];
			$data['description'] = '';
			if (!empty($originalData['description']) && $language->hasKey($data['description'])) {
				$data['description'] = '<p>' . $language->_($data['description'], '') . '</p>';
			}
			$data['description'] .= self::$lorem;
			$data['language'] = \count($this->languageCache) > 1 ? $code : '*';

			$data = $this->convertCustomFields($data, $code, $language);

			$data['associations'] = $newIds;

			$model = $this->factory->createModel('Event', 'Administrator', ['ignore_request' => true]);
			if (!$model->save($data)) {
				throw new \Exception(Text::_($model->getError()));
			}

			$item = $model->getItem();
			if (!\is_object($item)) {
				continue;
			}

			$newIds[$code] = $item->id;
		}

		$ids   = $app->getUserState('sampledata.dpcalendar.events', []);
		$ids[] = $newIds;
		$app->setUserState('sampledata.dpcalendar.events', $ids);
	}

	private function createLocation(array $data): int
	{
		$app = $this->getApplication();
		if (!$app instanceof CMSWebApplicationInterface) {
			return 0;
		}

		$data['state']    = 1;
		$data['access']   = (int)$app->get('access', 1);
		$data['language'] = '*';

		$model = $this->factory->createModel('Country', 'Administrator', ['ignore_request' => true]);
		if ($country = $model->getItem(['short_code' => $data['country']])) {
			$data['country'] = $country->id;
		}

		$model = $this->factory->createModel('Location', 'Administrator');
		if (!$model->save($data)) {
			throw new \Exception(Text::_($model->getError()));
		}

		$item = $model->getItem();
		if (!\is_object($item)) {
			return 0;
		}

		return (int)$item->id;
	}

	private function createCalendar(array $originalData): void
	{
		$app = $this->getApplication();
		if (!$app instanceof CMSWebApplicationInterface) {
			return;
		}

		if (empty($originalData['description'])) {
			$originalData['description'] = '';
		}
		if (empty($originalData['params'])) {
			$originalData['params'] = '';
		}

		$newIds = [];
		foreach ($this->languageCache as $code => $language) {
			$data = $originalData;

			$data['title'] = $language->_($data['title']) . (\count($this->languageCache) > 1 ? ' (' . $code . ')' : '');
			$alias         = ApplicationHelper::stringURLSafe($data['title']);

			// Set unicodeslugs if alias is empty
			if (trim(str_replace('-', '', $alias)) !== '' && trim(str_replace('-', '', $alias)) !== '0') {
				$unicode = $app->set('unicodeslugs', 1);
				$alias   = ApplicationHelper::stringURLSafe($data['title']);
				$app->set('unicodeslugs', $unicode);
			}

			$data['parent_id']       = 1;
			$data['id']              = 0;
			$data['published']       = 1;
			$data['access']          = (int)$app->get('access', 1);
			$data['created_user_id'] = $app->getIdentity() !== null ? $app->getIdentity()->id : 0;
			$data['extension']       = 'com_dpcalendar';
			$data['level']           = 1;
			$data['alias']           = $code . '-' . $alias;
			$data['associations']    = [];
			$data['language']        = \count($this->languageCache) > 1 ? $code : '*';

			$data = $this->convertCustomFields($data, $code, $language);

			$model = $app->bootComponent('categories')->getMVCFactory()->createModel('Category', 'Administrator');
			if (!$model->save($data)) {
				throw new \Exception($model->getError());
			}

			$newIds[$code] = $model->getItem()->id;
		}

		$ids   = $app->getUserState('sampledata.dpcalendar.calendars', []);
		$ids[] = $newIds;
		$app->setUserState('sampledata.dpcalendar.calendars', $ids);
	}

	private function createExternalCalendar(array $originalData): void
	{
		$app = $this->getApplication();
		if (!$app instanceof CMSWebApplicationInterface) {
			return;
		}

		$newIds = [];
		foreach ($this->languageCache as $code => $language) {
			$data = $originalData;

			$data['title']    = $language->_($data['title']) . (\count($this->languageCache) > 1 ? ' (' . $code . ')' : '');
			$data['language'] = \count($this->languageCache) > 1 ? $code : '*';
			$data['state']    = 1;

			$model = $this->factory->createModel('Extcalendar', 'Administrator', ['ignore_request' => true]);
			if (!$model->save($data)) {
				throw new \Exception(Text::_($model->getError()));
			}

			$item = $model->getItem();
			if (!\is_object($item)) {
				continue;
			}

			$newIds[$code] = ($originalData['plugin'] == 'ical' ? 'i-' : 'g-') . $item->id;
		}

		$ids   = $app->getUserState('sampledata.dpcalendar.calendars', []);
		$ids[] = $newIds;
		$app->setUserState('sampledata.dpcalendar.calendars', $ids);
	}

	private function createPrivateCalendar(array $originalData): void
	{
		$app = $this->getApplication();
		if (!$app instanceof CMSWebApplicationInterface) {
			return;
		}

		$data = $originalData;

		// We use here Text because we create only one calendar
		$data['displayname'] = Text::_($data['displayname']);

		$model = $this->factory->createModel('Davcalendar', 'Site', ['ignore_request' => true]);
		if (!$model->save($data)) {
			throw new \Exception(Text::_($model->getError()));
		}

		$item = $model->getItem();
		if (!\is_object($item)) {
			return;
		}

		$newIds = [];
		foreach (array_keys($this->languageCache) as $code) {
			$newIds[$code] = 'cd-' . $item->id;
		}

		$ids   = $app->getUserState('sampledata.dpcalendar.calendars', []);
		$ids[] = $newIds;
		$app->setUserState('sampledata.dpcalendar.calendars', $ids);
	}

	private function getEvent(int $id): \stdClass
	{
		$start = DPCalendarHelper::getDate('+1 day');
		$model = $this->factory->createModel('Events', 'Administrator', ['ignore_request' => true]);
		$model->setState('filter.children', $id);
		$model->setState('list.start-date', $start->format(DPCalendarHelper::getComponentParameter('event_form_date_format', 'd.m.Y')));
		$model->setState('list.limit', 1);

		$events = $model->getItems();
		if (!$events) {
			throw new \Exception('No event found!');
		}

		return reset($events);
	}

	private function setUp(): void
	{
		$app = $this->getApplication();
		if (!$app instanceof CMSWebApplicationInterface) {
			return;
		}

		$app->getLanguage()->load('com_dpcalendar', JPATH_ADMINISTRATOR . '/components/com_dpcalendar');
		foreach (LanguageHelper::getContentLanguages() as $language) {
			$lang = $this->languageFactory->createLanguage($language->lang_code);
			$lang->load('plg_sampledata_' . $this->_name, JPATH_PLUGINS . '/' . $this->_type . '/' . $this->_name);

			$this->languageCache[$language->lang_code] = $lang;
		}

		$this->factory = $app->bootComponent('dpcalendar')->getMVCFactory();

		// Disable mail
		$app->set('mailonline', false);
	}

	private function clearTable(string $name): void
	{
		$this->getDatabase()->setQuery('truncate #__dpcalendar_' . $name);
		$this->getDatabase()->execute();
	}

	/**
	 * @param $data
	 * @param $code
	 * @param $language
	 *
	 * @return mixed
	 */
	private function convertCustomFields(array $data, string $code, Language $language)
	{
		if (empty($data['fields'])) {
			return $data;
		}

		$data['com_fields'] = [];
		foreach ($data['fields'] as $fieldCode => $fields) {
			foreach ($fields as $field) {
				if ($fieldCode != $code) {
					continue;
				}

				$data['com_fields'][$field['name']] = $language->_($field['value']);
			}
		}
		unset($data['fields']);

		return $data;
	}
}
