<?php
/**
 * @package   DPCalendar
 * @copyright Copyright (C) 2022 Digital Peak GmbH. <https://www.digital-peak.com>
 * @license   https://www.gnu.org/licenses/gpl-3.0.html GNU/GPL
 */

namespace DigitalPeak\Plugin\DPAttachments\DPCalendar\Extension;

\defined('_JEXEC') or die();

use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\User\UserFactoryAwareTrait;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Event\Event;
use Joomla\Registry\Registry;

class DPCalendar extends CMSPlugin
{
	use DatabaseAwareTrait;
	use UserFactoryAwareTrait;

	protected $autoloadLanguage = true;

	public function onDPAttachmentsCheckPermission(Event $event): void
	{
		// When not event then ignore
		if ($event->getArgument('context') !== 'com_dpcalendar.event') {
			return;
		}

		// Don't edit action
		if ($event->getArgument('action') !== 'core.edit' && $event->getArgument('action') !== 'core.edit.state') {
			return;
		}

		// The global user to check
		$user = $this->getApplication() instanceof CMSApplicationInterface ? $this->getApplication()->getIdentity() : null;
		if ($user === null) {
			return;
		}

		// The database
		$db = $this->getDatabase();

		// Load the hosts
		$query = $db->getQuery(true);
		$query->select('e.created_by, e.catid, h.user_id')->from('#__dpcalendar_events_hosts h');
		$query->where('h.event_id = ' . (int)$event->getArgument('item_id'));
		$query->where('h.user_id = ' . (int)$user->id);
		$query->join('inner', '#__dpcalendar_events e', 'h.event_id = e.id');

		$db->setQuery($query);

		// Get the hosts
		$hosts = $db->loadObjectList();
		if (!$hosts) {
			return;
		}

		// When the current user is the author do nothing
		if ($user->id == $hosts[0]->created_by) {
			return;
		}

		// Check if there is a host which belongs to the current user
		foreach ($hosts as $host) {
			// The asset
			$asset  = 'com_dpcalendar.category.' . $host->catid;
			$author = $this->getUserFactory()->loadUserById($host->created_by);

			$allowed = $author->authorise($event->getArgument('action'), $asset);

			// If the edit action is requested we check for edit.own
			if (!$allowed
				&& $event->getArgument('action') === 'core.edit'
				&& $author->authorise('core.edit.own', $asset)
				&& $host->created_by == $author->id) {
				$allowed = true;
			}

			$event->setArgument('allowed', $allowed);
			$event->stopPropagation();
			return;
		}
	}

	public function onDPAttachmentsAfterRenderLayout(Event $event): void
	{
		if ($event->getArgument('name') !== 'attachments.render'
			|| $event->getArgument('data')['context'] !== 'com_dpcalendar.event') {
			return;
		}

		$options = clone $event->getArgument('data')['options'];

		$item = $options->get('item');
		if (!$item) {
			return;
		}

		$options->set('item', null);

		// Render the attachment form the original event as well
		if ($item->original_id > 0) {
			$event->setArgument('content', $event->getArgument('content') . $event->getArgument('component')->render(
				$event->getArgument('data')['context'],
				$item->original_id,
				$options,
				false
			));
		}
	}

	public function onDPAttachmentsBeforeProcessList(Event $event): void
	{
		if ($event->getArgument('context') !== 'com_dpcalendar.event') {
			return;
		}

		// When event admin, accept all
		if ($event->getArgument('component')->canDo('core.admin', 'com_dpcalendar.event', $event->getArgument('item_id'))) {
			return;
		}

		// When editor, accept all
		if ($event->getArgument('component')->canDo('core.edit', 'com_dpcalendar.event', $event->getArgument('item_id'))) {
			return;
		}


		$user = $this->getApplication() instanceof CMSApplicationInterface ? $this->getApplication()->getIdentity() : null;
		if ($user === null) {
			return;
		}

		$attachments = $event->getArgument('attachments');
		foreach ($attachments as $key => $attachment) {
			if (!$attachment->params->get('dpcalendar_event_ticket')) {
				continue;
			}

			$item = $event->getArgument('options')->get('item');
			if (!$item) {
				continue;
			}

			if (empty($item->tickets)) {
				unset($attachments[$key]);
				continue;
			}

			$found = false;
			foreach ($item->tickets as $ticket) {
				if ($user->id === 0) {
					continue;
				}
				if ($ticket->email !== $user->email && $ticket->user_id != $user->id) {
					continue;
				}
				$found = true;
			}

			if (!$found) {
				unset($attachments[$key]);
			}
		}

		$event->setArgument('attachments', $attachments);
	}

	public function onContentPrepareForm(Form $form, mixed $data): void
	{
		// The context
		$context = $form->getName();

		// When DPAttachments
		if ($context !== 'com_dpattachments.attachment') {
			return;
		}

		$dataContext = '';
		if (\is_object($data) && !empty($data->context)) {
			$dataContext = $data->context;
		}

		if (!$dataContext && \is_array($data) && !empty($data['context'])) {
			$dataContext = $data['context'];
		}

		$input = $this->getApplication() instanceof CMSApplicationInterface ? $this->getApplication()->getInput() : new Registry();

		$inputData = $input->get('jform', [], 'array');
		if (!$dataContext && $inputData && !empty($inputData['context'])) {
			$dataContext = $inputData['context'];
		}

		if ($dataContext === 'com_dpcalendar.event') {
			$form->loadFile(JPATH_PLUGINS . '/' . $this->_type . '/' . $this->_name . '/forms/event.xml');
		}
	}
}
