<?php
/**
 * @package   DPCalendar
 * @copyright Copyright (C) 2014 Digital Peak GmbH. <https://www.digital-peak.com>
 * @license   https://www.gnu.org/licenses/gpl-3.0.html GNU/GPL
 */

namespace DigitalPeak\Module\Upcoming\Site\Dispatcher;

\defined('_JEXEC') or die();

use DigitalPeak\Component\DPCalendar\Administrator\Helper\DateHelper;
use DigitalPeak\Component\DPCalendar\Administrator\Helper\DPCalendarHelper;
use DigitalPeak\Component\DPCalendar\Administrator\HTML\Document\HtmlDocument;
use DigitalPeak\Component\DPCalendar\Administrator\Router\Router;
use DigitalPeak\Component\DPCalendar\Administrator\Translator\Translator;
use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Dispatcher\AbstractModuleDispatcher;
use Joomla\CMS\HTML\Helpers\StringHelper;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Registry\Registry;

class Dispatcher extends AbstractModuleDispatcher
{
	protected function getLayoutData(): false|array
	{
		$data = parent::getLayoutData();
		if ($data === false) {
			return $data;
		}

		/** @var CMSApplicationInterface $app */
		$app = $data['app'];

		/** @var Registry $params */
		$params = $data['params'];

		// Helpers
		$document   = new HtmlDocument();
		$translator = new Translator();
		$dateHelper = new DateHelper();
		$dateHelper->setTranslator($translator);

		$layoutHelper = $app->bootComponent('dpcalendar')->getMVCFactory()->createModel('Layout', 'Administrator');
		$router       = new Router();

		$cParams = clone ComponentHelper::getParams('com_dpcalendar');
		$cParams->set('textbefore', '');
		$cParams->set('textafter', '');

		$moduleParams = $cParams->merge($params);

		// The display data
		$data = array_merge($data, [
			'document'     => $document,
			'layoutHelper' => $layoutHelper,
			'dateHelper'   => $dateHelper,
			'translator'   => $translator,
			'router'       => $router,
			'input'        => $app->getInput(),
			'params'       => $moduleParams,
			'moduleParams' => $moduleParams,
		]);

		$app->getLanguage()->load('com_dpcalendar', JPATH_ADMINISTRATOR . '/components/com_dpcalendar');
		$app->getLanguage()->load('com_dpcalendar', JPATH_SITE . '/components/com_dpcalendar');

		$model = $app->bootComponent('dpcalendar')->getMVCFactory()->createModel('Calendar', 'Site', ['ignore_request' => true]);
		$model->getState();
		$model->setState('filter.parentIds', $moduleParams->get('ids', ['root']));

		$ids = [];
		foreach ($model->getItems() as $calendar) {
			$ids[] = $calendar->getId();
		}

		if ($ids === []) {
			return $data;
		}

		$startDate = trim((string)$moduleParams->get('start_date', ''));
		if ($startDate === 'start of day') {
			$startDate = $dateHelper->getDate(null, true, 'UTC');
			$startDate->setTime(0, 0, 0);
		} else {
			$startDate = $dateHelper->getDate($startDate);
		}

		// Round to the last quarter
		$startDate->sub(new \DateInterval("PT" . $startDate->format("s") . "S"));
		$startDate->sub(new \DateInterval("PT" . ($startDate->format("i") % 15) . "M"));

		$endDate = trim((string)$moduleParams->get('end_date', ''));
		if ($endDate === 'same day') {
			$endDate = clone $startDate;
			$endDate->setTime(23, 59, 59);
		} elseif ($endDate !== '' && $endDate !== '0') {
			$tmp = $dateHelper->getDate($endDate);
			$tmp->sub(new \DateInterval("PT" . $tmp->format("s") . "S"));
			$tmp->sub(new \DateInterval("PT" . ($tmp->format("i") % 15) . "M"));
			$endDate = $tmp;
		} else {
			$endDate = null;
		}

		$model = $app->bootComponent('dpcalendar')->getMVCFactory()->createModel('Events', 'Site', ['ignore_request' => true]);
		$model->getState();
		$model->setState('list.limit', $moduleParams->get('max_events', 5));
		$model->setState('list.direction', $moduleParams->get('order', 'asc'));
		$model->setState('list.ordering', 'a.' . $moduleParams->get('sort', 'start_date'));
		$model->setState('category.id', $ids);
		$model->setState('category.recursive', true);
		$model->setState('filter.search', $moduleParams->get('filter', ''));
		$model->setState('filter.ongoing', $moduleParams->get('ongoing', 0));
		$model->setState('filter.expand', $moduleParams->get('expand', 1));
		$model->setState('filter.state', [1, 3]);
		$model->setState('filter.language', $app->getLanguage()->getTag());
		$model->setState('filter.publish_date', true);
		$model->setState('list.start-date', $startDate);
		$model->setState('list.end-date', $endDate);
		$model->setState('filter.author', $moduleParams->get('filter_author', 0));
		$model->setState('filter.featured', $moduleParams->get('filter_featured', 0));
		$model->setState('filter.tags', $moduleParams->get('filter_tags', []));

		$locationFilter = $moduleParams->get('filter_locations', []);
		if (\count($locationFilter) === 1 && $radius = $moduleParams->get('filter_radius', 0)) {
			$model->setState('filter.location', $app->bootComponent('dpcalendar')->getMVCFactory()->createModel('Location', 'Administrator', ['ignore_request' => true])->getItem($locationFilter[0]));
			$model->setState('filter.radius', $radius);
		} else {
			$model->setState('filter.locations', $locationFilter);
		}

		$events = $model->getItems();

		if (!$events && !$moduleParams->get('empty_text', 1)) {
			return false;
		}

		if ($moduleParams->get('sort', 'start_date') == 'start_date') {
			// Sort the array by user date
			usort($events, static function ($e1, $e2) use ($dateHelper, $moduleParams): int {
				$d1 = $dateHelper->getDate($e1->start_date, $e1->all_day);
				$d2 = $dateHelper->getDate($e2->start_date, $e2->all_day);
				if ($moduleParams->get('order', 'asc') !== 'asc') {
					$tmp = $d1;
					$d1  = $d2;
					$d2  = $tmp;
				}
				return strcmp($d1->format('c', true), $d2->format('c', true));
			});
		}

		PluginHelper::importPlugin('content');
		PluginHelper::importPlugin('dpcalendar');
		$now = $dateHelper->getDate();

		// The grouping option
		$grouping = $moduleParams->get('output_grouping', '');

		// The last computed heading
		$lastHeading = '';

		// The grouped events
		$groupedEvents = [];
		foreach ($events as $event) {
			$startDate    = $dateHelper->getDate($event->start_date, $event->all_day);
			$groupHeading = $grouping ? $startDate->format($grouping, true) : false;

			if ($groupHeading && $groupHeading !== $lastHeading) {
				$lastHeading = $groupHeading;
			}

			if (!\array_key_exists($lastHeading, $groupedEvents)) {
				$groupedEvents[$lastHeading] = [];
			}

			$groupedEvents[$lastHeading][] = $event;

			$event->text = $event->description ?: '';
			$app->triggerEvent('onContentPrepare', ['com_dpcalendar.event', &$event, &$event->params, 0]);
			$event->description = $event->text;

			if ($event->introText) {
				$event->text = $event->introText;
				$app->triggerEvent('onContentPrepare', ['com_dpcalendar.event', &$event, &$event->params, 0]);
				$event->introText = $event->text;
			}

			$event->realUrl = str_replace(
				['?tmpl=component', 'tmpl=component'],
				'',
				$router->getEventRoute($event->id, $event->catid, false, true, $moduleParams->get('default_menu_item', 0))
			);

			$desc = $params->get('description_length') === '0' ? '' : HTMLHelper::_('content.prepare', $event->description);
			if (!$event->introText && $desc && $params->get('description_length') > 0) {
				$event->introText = StringHelper::truncateComplex($desc, $params->get('description_length'));

				// Move the dots inside the last tag
				if (DPCalendarHelper::endsWith($event->introText, '...') && $pos = strrpos($event->introText, '</')) {
					$event->introText = trim(substr_replace($event->introText, '...</', $pos, 2), '.');
				}
			}

			if ($event->introText) {
				$event->alternative_readmore = $translator->translate('COM_DPCALENDAR_READ_MORE');

				// Meta data is handled differently
				$event->introText .= str_replace('itemprop="url"', '', (string)$layoutHelper->renderLayout(
					'joomla.content.readmore',
					[
						'item'   => $event,
						'params' => new Registry(['access-view' => true]),
						'link'   => $router->getEventRoute($event->id, $event->catid)
					]
				));
			}

			$event->truncatedDescription = $event->introText ?: $desc;

			// Determine if the event is running
			$date = $dateHelper->getDate($event->start_date);
			if (!empty($event->series_min_start_date) && !$moduleParams->get('expand', 1)) {
				$date = $dateHelper->getDate($event->series_min_start_date);
			}
			$event->ongoing_start_date = $date < $now ? $date : null;

			$date = $dateHelper->getDate($event->end_date);
			if (!empty($event->series_min_end_date) && !$moduleParams->get('expand', 1)) {
				$date = $dateHelper->getDate($event->series_min_end_date);
			}
			$event->ongoing_end_date = $date > $now ? $date : null;

			if ($moduleParams->get('show_display_events')) {
				$event->displayEvent = new \stdClass();
				$results             = $app->triggerEvent(
					'onContentAfterTitle',
					['com_dpcalendar.event', &$event, &$event->params, 0]
				);
				$event->displayEvent->afterDisplayTitle = trim(implode("\n", $results));

				$results = $app->triggerEvent(
					'onContentBeforeDisplay',
					['com_dpcalendar.event', &$event, &$event->params, 0]
				);
				$event->displayEvent->beforeDisplayContent = trim(implode("\n", $results));

				$results = $app->triggerEvent(
					'onContentAfterDisplay',
					['com_dpcalendar.event', &$event, &$event->params, 0]
				);
				$event->displayEvent->afterDisplayContent = trim(implode("\n", $results));
			}
		}

		$return = $app->getInput()->getInt('Itemid', 0);
		if (!empty($return)) {
			$return = $router->route('index.php?Itemid=' . $return);
		}

		$data['events']        = $events;
		$data['groupedEvents'] = $groupedEvents;
		$data['return']        = $return;
		$data['displayData']   = $data;

		return $data;
	}
}
