<?php
/**
 * @package   DPCalendar
 * @copyright Copyright (C) 2015 Digital Peak GmbH. <https://www.digital-peak.com>
 * @license   https://www.gnu.org/licenses/gpl-3.0.html GNU/GPL
 */

namespace DigitalPeak\Plugin\DPCalendar\RSEvents\Extension;

\defined('_JEXEC') or die();

use DigitalPeak\Component\DPCalendar\Administrator\Helper\DPCalendarHelper;
use DigitalPeak\Component\DPCalendar\Administrator\Plugin\DPCalendarPlugin;
use Joomla\CMS\Categories\Categories;
use Joomla\CMS\Date\Date;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Registry\Registry;

class RSEvents extends DPCalendarPlugin
{
	use DatabaseAwareTrait;

	protected string $identifier = 'rs';

	protected function getContent(string $calendarId, ?Date $startDate = null, ?Date $endDate = null, ?Registry $options = null): string
	{
		if (!is_dir(JPATH_ADMINISTRATOR . '/components/com_rseventspro')) {
			return '';
		}

		$db = $this->getDatabase();

		$query = "select
				e.*,
				c.title as category,
				l.name as location_name, l.description as location_description, l.address as location_address, l.coordinates as location_coordinates
			  from `#__rseventspro_events` as e
			  inner join `#__rseventspro_taxonomy` as t on e.id=t.ide
			  inner join `#__categories` as c on c.id=t.id
			  left join `#__rseventspro_locations` as l on l.id=e.location
			  where t.id=" . (int)$calendarId;

		$dateCondition = '';
		if ($startDate instanceof Date) {
			$startDate     = $db->quote($startDate->toSql());
			$dateCondition = 'e.start  >= ' . $startDate;
			if ($endDate instanceof Date) {
				$endDate = $db->quote($endDate->toSql());

				// Between start and end date
				$dateCondition = 'e.end between ' . $startDate . ' and ' . $endDate . ' or e.start between ' . $startDate . ' and ' . $endDate .
					' or (e.start < ' . $startDate . ' and e.end > ' . $endDate . ') ';

				// If it is a recurring event, repeating end needs to be checked
				$dateCondition .= 'or (e.recurring = 1 and e.repeat_end between ' . $startDate . ' and ' . $endDate;
				$dateCondition .= ' or (e.start < ' . $startDate . ' and e.repeat_end > ' . $endDate . ')) ';
			}
		} elseif ($endDate instanceof Date) {
			$dateCondition = '(e.start  <= ' . $endDate . ' or (e.recurring = 1 and e.repeat_end <= ' . $endDate . '))';
		}
		if ($dateCondition !== '' && $dateCondition !== '0') {
			$query .= ' and (' . $dateCondition . ')';
		}
		$db->setQuery($query);
		$events = $db->loadObjectList();

		$text   = [];
		$text[] = 'BEGIN:VCALENDAR';
		if (!empty($events)) {
			foreach ($events as $event) {
				$text[] = 'BEGIN:VEVENT';
				$text[] = 'UID:' . md5($event->id . 'RSEventsPro');
				$text[] = 'CATEGORIES:' . $event->category;
				$text[] = 'SUMMARY:' . $event->name;
				$text[] = 'DESCRIPTION:' . strip_tags($this->replaceNl($event->description));
				$text[] = 'X-ALT-DESC;FMTTYPE=text/html:' . $this->replaceNl($event->description);
				$text[] = 'X-HITS:' . $event->hits;
				$text[] = 'X-URL:' . $event->URL;

				// start
				if ($event->allday) {
					$text[] = 'DTSTART;VALUE=DATE:' . DPCalendarHelper::getDate($event->start, true)->format('Ymd');
				} else {
					$text[] = 'DTSTART:' . DPCalendarHelper::getDate($event->start, false)->format('Ymd\THis\Z');
				}

				// end
				if ($event->allday) {
					$end = DPCalendarHelper::getDate($event->start, true);
					$end->modify('+1 day');
					$text[] = 'DTEND;VALUE=DATE:' . $end->format('Ymd');
				} else {
					$text[] = 'DTEND:' . DPCalendarHelper::getDate($event->end, false)->format('Ymd\THis\Z');
				}

				// recurring event?
				if ($event->recurring) {
					$text[] = $this->createRRule($event);
				}

				// add location + geo (if available)
				if (!\is_null($event->location_name)) {
					$text[] = 'LOCATION:' . $event->location_name . (\strlen((string)$event->location_address) !== 0 ? ' (' . $event->location_address . ')' : '');
					if (!\is_null($event->location_coordinates)) {
						$text[] = 'GEO:' . str_replace(",", ";", (string)$event->location_coordinates);
					}
				}

				$text[] = 'END:VEVENT';
			}
		}
		$text[] = 'END:VCALENDAR';

		return implode(PHP_EOL, $text);
	}

	protected function fetchCalendars(array $calendarIds = []): array
	{
		if (!is_dir(JPATH_ADMINISTRATOR . '/components/com_rseventspro')) {
			return [];
		}

		$root = Categories::getInstance('RSEventsPro');
		if (!$root instanceof Categories) {
			return [];
		}
		$root = $root->get('root');

		if (empty($root)) {
			return [];
		}

		$calendars = [];
		foreach ($root->getChildren(true) as $calendar) {
			if ($calendarIds !== [] && !\in_array($calendar->id, $calendarIds)) {
				continue;
			}
			$calendars[] = $this->createCalendar((string)$calendar->id, $calendar->title, $calendar->description);
		}

		return $calendars;
	}

	protected function createRRule($event): string
	{
		$text = 'RRULE:FREQ=';

		switch ($event->repeat_type) {
			default:
			case 1:
				$text .= 'DAILY;';
				break;
			case 2:
				$text .= 'WEEKLY;';
				break;
			case 3:
				$text .= 'MONTHLY;';
				break;
			case 4:
				$text .= 'YEARLY;';
				break;
		}

		// could be zero
		$text .= 'INTERVAL=' . $event->repeat_interval . ';';

		if ($event->repeat_end != $this->getDatabase()->getNullDate()) {
			$text .= 'UNTIL=' . DPCalendarHelper::getDate($event->repeat_end, false)->format('Ymd\THis\Z') . ';';
		}

		$dayMap = [
			0 => 'SU',
			1 => 'MO',
			2 => 'TU',
			3 => 'WE',
			4 => 'TH',
			5 => 'FR',
			6 => 'SA'
		];
		switch ($event->repeat_type) {
			default:
			case 1:
			case 2:
				require_once JPATH_SITE . '/components/com_rseventspro/helpers/events.php';

				$rsEventHelper = \RSEvent::getInstance($event->id);
				$text .= 'BYDAY=' . implode(
					',',
					array_map(
						static fn ($x): string => $dayMap[$x],
						$rsEventHelper->repeatEventDays()
					)
				);
				$text .= ';';
				break;
			case 3:
				if ($event->repeat_on_type == 1) {
					$text .= 'BYMONTHDAY=' . $event->repeat_on_day;
				}
				if ($event->repeat_on_type == 2) {
					$text .= 'BYDAY=' . $event->repeat_on_day_order . $dayMap[$event->repeat_on_day_type];
				}
				break;
		}

		return $text;
	}
}
