<?php
/**
 * @package    DPCalendar
 * @author     Digital Peak http://www.digital-peak.com
 * @copyright  Copyright (C) 2007 - 2013 Digital Peak. All rights reserved.
 * @license    http://www.gnu.org/licenses/gpl.html GNU/GPL
 */
namespace Sabre\CalDAV\Backend;
use Sabre\VObject;
use Sabre\CalDAV;
use Sabre\DAV;
\JLoader::import('components.com_dpcalendar.helpers.dpcalendar', JPATH_ADMINISTRATOR);
\JLoader::import('components.com_dpcalendar.helpers.ical', JPATH_ADMINISTRATOR);

class DPCalendar extends PDO
{

	public function getCalendarsForUser ($principalUri)
	{
		$calendars = parent::getCalendarsForUser($principalUri);

		$levels = \JAccess::getAuthorisedViewLevels(\JUserHelper::getUserId(str_replace('principals/', '', $principalUri)));

		$user = \JFactory::getUser();
		foreach (\DPCalendarHelper::getCalendar('root')->getChildren() as $calendar)
		{
			if (! in_array($calendar->access, $levels))
			{
				continue;
			}

			$writePermission = $user->authorise('core.edit', 'com_dpcalendar.category.' . $calendar->id) &&
					 $user->authorise('core.delete', 'com_dpcalendar.category.' . $calendar->id);

			$calendars[] = array(
					'id' => 'dp-' . $calendar->id,
					'uri' => 'dp-' . $calendar->id,
					'principaluri' => $principalUri,
					'{http://calendarserver.org/ns/}shared-url' => 'dp-' . $calendar->id,
					'{http://sabredav.org/ns}owner-principal' => 'principals/dpcalendar-calendar',
					'{http://sabredav.org/ns}read-only' => $writePermission,
					'{' . CalDAV\Plugin::NS_CALDAV . '}supported-calendar-component-set' => new CalDAV\Property\SupportedCalendarComponentSet(
							array(
									'VEVENT'
							))
			);
		}

		return $calendars;
	}

	public function getCalendarObject ($calendarId, $objectUri)
	{
		if (strpos($objectUri, 'dpcalendar-event-') !== false)
		{
			$event = $this->getTable();
			$event->load(str_replace(array(
					'dpcalendar-event-',
					'.ics'
			), '', $objectUri));

			if (empty($event))
			{
				return null;
			}

			return $this->toSabreArray($event);
		}
		return parent::getCalendarObject($calendarId, $objectUri);
	}

	public function getCalendarObjects ($calendarId)
	{
		if (strpos($calendarId, 'dp-') !== false)
		{
			\JModelLegacy::addIncludePath(JPATH_SITE . DS . 'components' . DS . 'com_dpcalendar' . DS . 'models', 'DPCalendarModel');

			$model = \JModelLegacy::getInstance('Events', 'DPCalendarModel');
			$model->getState();
			$model->setState('list.limit', 1000);
			$model->setState('category.id', str_replace('dp-', '', $calendarId));
			$model->setState('category.recursive', true);
			$model->setState('filter.ongoing', 1);

			$model->setState('list.start-date', 0);
			$model->setState('list.end-date', \DPCalendarHelper::getDate(self::MAX_DATE)->format('U'));

			$data = array();
			foreach ($model->getItems() as $event)
			{
				$id = 'dpcalendar-event-' . $event->id;
				if ($event->original_id > 0)
				{
					$id = 'dpcalendar-event-' . $event->original_id;
				}
				if (key_exists($id, $data))
				{
					continue;
				}
				if ($event->original_id > 0)
				{
					$original = $this->getTable();
					$original->load($event->original_id);
					$event = $original;
				}
				$data[$id] = $this->toSabreArray($event);
			}
			return $data;
		}

		return parent::getCalendarObjects($calendarId);
	}

	public function calendarQuery ($calendarId, array $filters)
	{
		if (strpos($calendarId, 'dp-') !== false)
		{
			$timeRange = array();
			if (count($filters['comp-filters']) > 0 && ! $filters['comp-filters'][0]['is-not-defined'])
			{
				$componentType = $filters['comp-filters'][0]['name'];

				if ($componentType == 'VEVENT' && isset($filters['comp-filters'][0]['time-range']))
				{
					$timeRange = $filters['comp-filters'][0]['time-range'];
				}
			}

			\JModelLegacy::addIncludePath(JPATH_SITE . DS . 'components' . DS . 'com_dpcalendar' . DS . 'models', 'DPCalendarModel');

			$model = \JModelLegacy::getInstance('Events', 'DPCalendarModel');
			$model->getState();
			$model->setState('list.limit', 1000);
			$model->setState('category.id', str_replace('dp-', '', $calendarId));
			$model->setState('category.recursive', true);
			$model->setState('filter.ongoing', 1);

			if (key_exists('start', $timeRange))
			{
				$model->setState('list.start-date', $timeRange['start']->getTimeStamp());
			}
			if (key_exists('end', $timeRange))
			{
				$model->setState('list.end-date', $timeRange['end']->getTimeStamp());
			}

			$data = array();
			foreach ($model->getItems() as $event)
			{
				$id = 'dpcalendar-event-' . $event->id;
				if ($event->original_id > 0)
				{
					$id = 'dpcalendar-event-' . $event->original_id;
				}
				if (! $this->validateFilterForObject(array(
						'uri' => $id,
						'calendarid' => $calendarId
				), $filters))
				{
					continue;
				}
				$data[$id] = $id;
			}
			return $data;
		}
		return parent::calendarQuery($calendarId, $filters);
	}

	public function createCalendarObject ($calendarId, $objectUri, $calendarData)
	{
		if (strpos($calendarId, 'dp-') !== false)
		{
			$event = $this->getTable();
			$vEvent = VObject\Reader::read($calendarData)->VEVENT;
			$event->alias = \JApplication::stringURLSafe($vEvent->SUMMARY->value);
			$event->catid = str_replace('dp-', '', $calendarId);
			$event->state = 1;

			$this->merge($event, $vEvent);
			return null;
		}
		return parent::createCalendarObject($calendarId, $objectUri, $calendarData);
	}

	public function updateCalendarObject ($calendarId, $objectUri, $calendarData)
	{
		if (strpos($calendarId, 'dp-') !== false)
		{
			$event = $this->getTable();
			$event->load(str_replace(array(
					'dpcalendar-event-',
					'.ics'
			), '', $objectUri));
			$obj = VObject\Reader::read($calendarData);

			if ($event->original_id == '-1')
			{
				foreach ($obj->VEVENT as $vEvent)
				{
					if (! empty($vEvent->RRULE->value))
					{
						$this->merge($event, $vEvent);
					}
				}

				$db = \JFactory::getDbo();
				$db->setQuery('select * from #__dpcalendar_events where original_id = ' . $db->quote($event->id));
				$children = $db->loadObjectList('', 'DPCalendarTableEvent');

				foreach ($obj->VEVENT as $vEvent)
				{
					if (! isset($vEvent->{'RECURRENCE-ID'}))
					{
						continue;
					}
					$startDate = (string) $vEvent->{'RECURRENCE-ID'}->value;

					foreach ($children as $child)
					{
						if ($child->recurrence_id == $startDate)
						{
							$this->merge($child, $vEvent);
							break;
						}
					}
				}
			}
			else
			{
				$this->merge($event, $obj->VEVENT);
			}

			return null;
		}
		return parent::updateCalendarObject($calendarId, $objectUri, $calendarData);
	}

	public function deleteCalendarObject ($calendarId, $objectUri)
	{
		if (strpos($calendarId, 'dp-') !== false)
		{
			$event = $this->getTable();
			$event->delete(str_replace(array(
					'dpcalendar-event-',
					'.ics'
			), '', $objectUri));
		}

		return parent::deleteCalendarObject($calendarId, $objectUri);
	}

	private function getTable ($type = 'Event')
	{
		\JTable::addIncludePath(JPATH_ADMINISTRATOR . DS . 'components' . DS . 'com_dpcalendar' . DS . 'tables');

		return \JTable::getInstance($type, 'DPCalendarTable');
	}

	private function merge ($dpEvent, $vEvent)
	{
		if (isset($vEvent->SUMMARY))
		{
			$dpEvent->title = $vEvent->SUMMARY->value;
		}
		else
		{
			$dpEvent->title = '(no title)';
		}

		if (isset($vEvent->DESCRIPTION))
		{
			$dpEvent->description = $vEvent->DESCRIPTION->value;
		}
		$dpEvent->all_day = strlen($vEvent->DTSTART->value) > 10 ? 0 : 1;

		$start = $vEvent->DTSTART->getDateTime();
		if ($dpEvent->all_day)
		{
			$start->setTime(0, 0, 0);
		}
		else
		{
			$start->setTimezone(new \DateTimeZone('UTC'));
		}
		$dpEvent->start_date = $start->format(\JFactory::getDbo()->getDateFormat());

		$end = $vEvent->DTEND->getDateTime();
		if ($dpEvent->all_day)
		{
			$end->setTime(0, 0, 0);
			$end->modify('-1 day');
		}
		else
		{
			$end->setTimezone(new \DateTimeZone('UTC'));
		}
		$dpEvent->end_date = $end->format(\JFactory::getDbo()->getDateFormat());

		if (isset($vEvent->{'X-ALT-DESC'}) && ! empty($vEvent->{'X-ALT-DESC'}->value))
		{
			$dpEvent->description = $vEvent->{'X-ALT-DESC'}->value;
		}
		if (isset($vEvent->{'X-COLOR'}) && ! empty($vEvent->{'X-COLOR'}->value))
		{
			$dpEvent->color = $vEvent->{'X-COLOR'}->value;
		}
		if (isset($vEvent->{'X-URL'}) && ! empty($vEvent->{'X-URL'}->value))
		{
			$dpEvent->url = $vEvent->{'X-URL'}->value;
		}
		if (isset($vEvent->RRULE) && ! empty($vEvent->RRULE->value))
		{
			$dpEvent->rrule = $vEvent->RRULE->value;
		}
		if (isset($vEvent->LOCATION) && ! empty($vEvent->LOCATION->value))
		{
			$locationString = $vEvent->LOCATION->value;

			// The ical creator escapes , and ; so we need to turn them back
			$locationString = str_replace('\,', ',', $locationString);
			$locationString = str_replace('\;', ';', $locationString);

			\JModelLegacy::addIncludePath(JPATH_ADMINISTRATOR . DS . 'components' . DS . 'com_dpcalendar' . DS . 'models', 'DPCalendarModel');
			$location = null;

			if (isset($vEvent->GEO) && ! empty($vEvent->GEO->value))
			{
				$parts = explode(';', $vEvent->GEO->value);
				if (count($parts) == 2)
				{
					$model = \JModelLegacy::getInstance('Locations', 'DPCalendarModel', array(
							'ignore_request' => true
					));
					$model->getState();
					$model->setState('list.limit', 1);
					$model->setState('filter.latitude', $parts[0]);
					$model->setState('filter.longitude', $parts[1]);

					$locations = $model->getItems();
					if (! empty($locations))
					{
						$location = reset($locations);
					}
				}
			}

			if (! $location)
			{
				$model = \JModelLegacy::getInstance('Locations', 'DPCalendarModel');
				$model->getState();
				$model->setState('list.limit', 10000);

				$locations = $model->getItems();

				foreach ($locations as $l)
				{
					if (\DPCalendarHelperLocation::format($l) == $locationString)
					{
						$location = $l;
						break;
					}
				}
				if (! $location)
				{
					$newLocation = \DPCalendarHelperLocation::get($locationString);
					$newLocation->id = 0;
					$table = $this->getTable('Location');
					$table->bind((array) $newLocation);
					if ($table->store())
					{
						$newLocation->id = $table->id;
					}

					$location = $newLocation;
				}
			}
			if ($location)
			{
				$dpEvent->location_ids = array(
						$location->id
				);
			}
		}

		\JModelLegacy::addIncludePath(JPATH_SITE . DS . 'components' . DS . 'com_dpcalendar' . DS . 'models', 'DPCalendarModel');

		$model = \JModelLegacy::getInstance('Form', 'DPCalendarModel');
		$model->save((array) $dpEvent);

		if ($model->getError())
		{
			throw new \Sabre\DAV\Exception\BadRequest('Error happened storing the event: ' . $model->getError());
		}
	}

	private function toSabreArray ($event)
	{
		$events = array();
		$event->locations = $this->getLocations($event);
		if ($event->original_id == '-1')
		{
			$event->uid = 'dpcalendar-event-' . $event->id;
			$events[] = $event;
			$db = \JFactory::getDbo();
			$db->setQuery(
					'select * from #__dpcalendar_events where original_id = ' . $db->quote($event->id) . ' and ( modified != ' .
							 $db->quote($db->getNullDate()) . ' and modified != ' . $db->quote($event->modified) . ')');
			$children = $db->loadObjectList('', 'DPCalendarTableEvent');
			foreach ($children as $child)
			{
				$child->uid = 'dpcalendar-event-' . $event->id;
				$child->locations = $this->getLocations($child);
				$events[] = $child;
			}
		}
		else
		{
			$event->uid = 'dpcalendar-event-' . $event->id;
			$events[] = $event;
		}
		$ical = \DPCalendarHelperIcal::createIcalFromEvents($events);
		$data = array(
				'id' => $event->id,
				'uri' => $event->uid . '.ics',
				'lastmodified' => \DPCalendarHelper::getDate($event->modified)->format('U'),
				'etag' => md5($ical),
				'calendarid' => 'dp-' . $event->catid,
				'size' => strlen($ical),
				'calendardata' => $ical
		);
		return $data;
	}

	private function getLocations ($event)
	{
		$db = \JFactory::getDbo();
		$query = $db->getQuery(true);
		$query->from('#__dpcalendar_locations l');
		$query->select("*");
		$query->join('right', '#__dpcalendar_events_location AS rel ON l.id = rel.location_id');
		$query->where('rel.event_id = ' . $event->id);
		$query->group('l.id');
		$db->setQuery($query);
		return $db->loadObjectList();
	}

	private function log ($message)
	{
		\JLog::add(print_r($message, true), \JLog::ERROR, 'com_dpcalendar');
	}
}
