<?php
/**
 * @package   DPCalendar
 * @copyright Copyright (C) 2014 Digital Peak GmbH. <https://www.digital-peak.com>
 * @license   https://www.gnu.org/licenses/gpl-3.0.html GNU/GPL
 */

namespace DigitalPeak\Plugin\Finder\DPCalendar\Extension;

defined('_JEXEC') or die();

use DigitalPeak\Component\DPCalendar\Site\Helper\RouteHelper;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Table\Table;
use Joomla\Component\Finder\Administrator\Indexer\Adapter;
use Joomla\Component\Finder\Administrator\Indexer\Helper;
use Joomla\Component\Finder\Administrator\Indexer\Indexer;
use Joomla\Component\Finder\Administrator\Indexer\Result;
use Joomla\Database\DatabaseQuery;
use Joomla\Registry\Registry;

class DPCalendar extends Adapter
{
	protected $context          = 'DPCalendar';
	protected $extension        = 'com_dpcalendar';
	protected $layout           = 'event';
	protected $type_title       = 'Event';
	protected $table            = '#__dpcalendar_events';
	protected $autoloadLanguage = true;

	public function onFinderCategoryChangeState(string $extension, array $pks, int $value): void
	{
		if ($extension === 'com_dpcalendar') {
			$this->categoryStateChange($pks, $value);
		}
	}

	/**
	 * @phpstan-param Table&object{id: string, link_id: string} $table
	 */
	public function onFinderAfterDelete(string $context, $table): bool
	{
		if ($context === 'com_dpcalendar.event') {
			return $this->remove($table->id);
		}

		if ($context === 'com_finder.index') {
			return $this->remove($table->link_id);
		}

		return true;
	}

	/**
	 * @phpstan-param Table&object{id: integer, access: integer} $row
	 */
	public function onFinderAfterSave(string $context, $row, bool $isNew): bool
	{
		if ($context !== 'com_dpcalendar.event' && $context !== 'com_dpcalendar.form') {
			return true;
		}

		if (!$isNew && $this->old_access != $row->access) {
			$this->itemAccessChange($row);
		}
		$this->reindex($row->id);

		return true;
	}

	/**
	 * @param Table $row
	 */
	public function onFinderBeforeSave(string $context, $row, bool $isNew): bool
	{
		if ($context !== 'com_dpcalendar.event' && $context !== 'com_dpcalendar.form') {
			return true;
		}

		if (!$isNew) {
			$this->checkItemAccess($row);
		}

		return true;
	}

	public function onFinderChangeState(string $context, array $pks, int $value): void
	{
		if ($context === 'com_dpcalendar.event' || $context === 'com_dpcalendar.form') {
			$this->itemStateChange($pks, $value);
		}

		if ($context !== 'com_plugins.plugin') {
			return;
		}

		if ($value !== 0) {
			return;
		}

		$this->pluginDisable($pks);
	}

	protected function index(Result $item, string $format = 'html'): bool
	{
		if (ComponentHelper::isEnabled($this->extension) === false) {
			return true;
		}

		$item->setLanguage();

		$registry = new Registry();
		$registry->loadString($item->params ?? '');

		$item->params = ComponentHelper::getParams('com_dpcalendar', true);
		$item->params->merge($registry);

		$registry = new Registry();
		$registry->loadString($item->metadata ?? '');

		$item->metadata = $registry;

		// Render the title from the layout
		$path = PluginHelper::getLayoutPath('finder', 'dpcalendar', 'title');
		ob_start();
		include $path;
		$item->title = ob_get_clean() ?: '';

		// Render the description from the layout
		$path = PluginHelper::getLayoutPath('finder', 'dpcalendar', 'description');
		ob_start();
		include $path;
		$item->summary = ob_get_clean() ?: '';

		// Unset here description and body so no duplicate content is shown
		$item->description = '';
		$item->body        = '';

		$item->url   = $this->getURL($item->id, $this->extension, $this->layout);
		$item->route = RouteHelper::getEventRoute($item->id, $item->catid, false, false);
		$item->route = str_replace(['tmpl=component', 'tmpl=raw'], ['', ''], $item->route);

		$title = $this->getItemMenuTitle($item->url);

		// Adjust the title if necessary
		if (!empty($title) && $this->params->get('use_menu_title', true)) {
			$item->title = $title;
		}
		// Add the meta-author
		$item->metaauthor = $item->metadata->get('author');

		// Add the meta-data processing instructions.
		$item->addInstruction((string)Indexer::META_CONTEXT, 'metakey');
		$item->addInstruction((string)Indexer::META_CONTEXT, 'metadesc');
		$item->addInstruction((string)Indexer::META_CONTEXT, 'metaauthor');
		$item->addInstruction((string)Indexer::META_CONTEXT, 'author');
		$item->addInstruction((string)Indexer::META_CONTEXT, 'created_by_alias');

		$item->state = $this->translateState($item->state);
		$item->addTaxonomy('Type', 'Dpcalendar');
		$item->addTaxonomy('Language', $item->language);
		Helper::getContentExtras($item);
		$this->indexer->index($item);

		return true;
	}

	protected function setup(): bool
	{
		return true;
	}

	protected function getListQuery($query = null)
	{
		// Check if we can use the supplied SQL query.
		$query = $query instanceof DatabaseQuery ? $query : $this->db->getQuery(true)
			->select('a.*')
			->select('a.publish_up AS publish_start_date, a.publish_down AS publish_end_date')
			->select('c.title AS category, c.published AS cat_state, c.access AS cat_access');

		// Handle the alias CASE WHEN portion of the query
		$case_when_item_alias = ' CASE WHEN ';
		$case_when_item_alias .= $query->charLength('a.alias', '!=', '0');
		$case_when_item_alias .= ' THEN ';
		$a_id = $query->castAs('CHAR', 'a.id');
		$case_when_item_alias .= $query->concatenate([$a_id, 'a.alias'], ':');
		$case_when_item_alias .= ' ELSE ';
		$case_when_item_alias .= $a_id . ' END as slug';
		$query->select($case_when_item_alias);

		$case_when_category_alias = ' CASE WHEN ';
		$case_when_category_alias .= $query->charLength('c.alias', '!=', '0');
		$case_when_category_alias .= ' THEN ';
		$c_id = $query->castAs('CHAR', 'c.id');
		$case_when_category_alias .= $query->concatenate([$c_id, 'c.alias'], ':');
		$case_when_category_alias .= ' ELSE ';
		$case_when_category_alias .= $c_id . ' END as catslug';
		$query->select($case_when_category_alias)
			->select('u.name AS author')
			->from('#__dpcalendar_events AS a')
			->join('LEFT', '#__categories AS c ON c.id = a.catid')
			->join('LEFT', '#__users AS u ON u.id = a.created_by');

		return $query;
	}
}
