<?php
/**
 * @package   DPMedia
 * @copyright Copyright (C) 2022 Digital Peak GmbH. <https://www.digital-peak.com>
 * @license   https://www.gnu.org/licenses/gpl-3.0.html GNU/GPL
 */

namespace DigitalPeak\Plugin\Filesystem\DPPermissions\Extension;

use Joomla\CMS\Access\Access;
use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Application\CMSWebApplicationInterface;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Response\JsonResponse;
use Joomla\CMS\Session\Session;
use Joomla\Component\Media\Administrator\Event\FetchMediaItemEvent;
use Joomla\Component\Media\Administrator\Event\FetchMediaItemsEvent;
use Joomla\Component\Media\Administrator\Event\MediaProviderEvent;
use Joomla\Component\Users\Administrator\Helper\UsersHelper;
use Joomla\Database\DatabaseAwareInterface;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Database\ParameterType;
use Joomla\Event\SubscriberInterface;
use Joomla\Filesystem\Path;

class Permissions extends CMSPlugin implements SubscriberInterface, DatabaseAwareInterface
{
	use DatabaseAwareTrait;

	public static function getSubscribedEvents(): array
	{
		return [
			'onSetupProviders'               => 'setupProviders',
			'onFetchMediaItems'              => 'adaptFiles',
			'onFetchMediaItem'               => 'adaptFile',
			'onAjaxSaveGroupsPathPermission' => 'saveGroups',
			'onAjaxGetGroupsPathPermissions' => 'getGroups'
		];
	}

	protected $autoloadLanguage = true;

	public function setupProviders(MediaProviderEvent $event): void
	{
		$app = $this->getApplication();
		if (!$app instanceof CMSWebApplicationInterface || !$this->canSetPermissions()) {
			return;
		}

		HTMLHelper::_('behavior.core');
		HTMLHelper::_('script', 'plg_filesystem_dppermissions/plugin/dppermissions.min.js', ['relative' => true, 'version' => 'auto']);
		HTMLHelper::_('stylesheet', 'plg_filesystem_dppermissions/plugin/dppermissions.min.css', ['relative' => true, 'version' => 'auto']);

		$app->getDocument()->addScriptOptions('DPPermissions.groups', UsersHelper::getGroups());

		Text::script('LIB_DPMEDIA_TEXT_CLOSE');
		Text::script('PLG_FILESYSTEM_DPPERMISSIONS_TEXT_PERMISSIONS');
		Text::script('PLG_FILESYSTEM_DPPERMISSIONS_TEXT_PERMISSIONS_HEADER');
		Text::script('PLG_FILESYSTEM_DPPERMISSIONS_TEXT_PERMISSIONS_DESC');
	}

	public function adaptFiles(FetchMediaItemsEvent $event): void
	{
		$app = $this->getApplication();
		if (!$app instanceof CMSApplicationInterface) {
			return;
		}

		$user = $app->getIdentity();
		if ($user === null) {
			return;
		}

		// When global admin then return all to prevent lock out
		if ($user->authorise('core.admin')) {
			return;
		}

		$items = $event->getArgument('items');
		if (!$items) {
			return;
		}

		$dirs = [];
		foreach ($items as $item) {
			$parent = dirname($item->path);
			while ($item->adapter . ':' !== $parent) {
				$dirs[$parent] = new \stdClass();
				$parent        = dirname($parent);
			}

			if ($item->type !== 'dir') {
				continue;
			}

			$dirs[$item->path] = $item;
		}

		if ($dirs === []) {
			return;
		}

		$query = $this->getDatabase()->getQuery(true);
		$query->select('path, group_id')->from('#__dppermissions')->where('path in (' . implode(',', $query->bindArray(array_keys($dirs))) . ')');

		$this->getDatabase()->setQuery($query);

		$permissions = [];
		foreach ($this->getDatabase()->loadObjectList() as $permission) {
			if (!array_key_exists($permission->path, $permissions)) {
				$permissions[$permission->path] = [];
			}
			$permissions[$permission->path][] = $permission->group_id;
		}

		$userGroups = Access::getGroupsByUser($user->id);
		foreach ($permissions as $path => $groups) {
			foreach ($items as $index => $item) {
				if (strpos($item->path, (string) $path) !== 0) {
					continue;
				}
				if (array_intersect($groups, $userGroups) !== []) {
					continue;
				}
				unset($items[$index]);
			}
		}

		$event->setArgument('items', $items);
	}

	public function adaptFile(FetchMediaItemEvent $event): void
	{
		$app = $this->getApplication();
		if (!$app instanceof CMSApplicationInterface) {
			return;
		}

		$user = $app->getIdentity();
		if ($user === null) {
			return;
		}

		// When global admin then return all to prevent lock out
		if ($user->authorise('core.admin')) {
			return;
		}

		$item = $event->getArgument('item');
		if (!$item) {
			return;
		}

		$dirs   = [$item->path => $item];
		$parent = dirname($item->path);
		while ($item->adapter . ':' !== $parent) {
			$dirs[$parent] = new \stdClass();
			$parent        = dirname($parent);
		}

		$query = $this->getDatabase()->getQuery(true);
		$query->select('path, group_id')->from('#__dppermissions')->where('path in (' . implode(',', $query->bindArray(array_keys($dirs))) . ')');

		$this->getDatabase()->setQuery($query);

		$permissions = [];
		foreach ($this->getDatabase()->loadObjectList() as $permission) {
			if (!array_key_exists($permission->path, $permissions)) {
				$permissions[$permission->path] = [];
			}
			$permissions[$permission->path][] = $permission->group_id;
		}

		$userGroups = Access::getGroupsByUser($user->id);
		foreach ($permissions as $path => $groups) {
			if (strpos($item->path, (string) $path) !== 0) {
				continue;
			}
			if (array_intersect($groups, $userGroups) !== []) {
				continue;
			}
			$event->removeArgument('item');
		}
	}

	public function saveGroups(): void
	{
		Session::checkToken('get') or die(Text::_('JINVALID_TOKEN'));

		$app = $this->getApplication();
		if (!$app instanceof CMSApplicationInterface) {
			return;
		}

		if (!$this->canSetPermissions()) {
			throw new \Exception('Not allowed', 403);
		}

		$path = $app->getInput()->json->getString('path', '');
		if ($path === '' || $path === '0') {
			return;
		}
		$path = str_replace('//', '/', Path::clean($path, '/'));

		$query = $this->getDatabase()->getQuery(true);
		$query->delete('#__dppermissions')->where('path=:path')->bind(':path', $path);

		$this->getDatabase()->setQuery($query)->execute();

		$groups = (array)$app->getInput()->json->get('groups', '', 'ARRAY');
		if ($groups === []) {
			echo new JsonResponse($groups, $app->getLanguage()->_('PLG_FILESYSTEM_DPPERMISSIONS_TEXT_SUCCESSFULLY_SAVED_GROUPS'));
			$app->close();
			return;
		}

		$query = $this->getDatabase()->getQuery(true)->insert('#__dppermissions')->columns('path, group_id');
		foreach ($groups as $groupId) {
			$query->values(implode(',', $query->bindArray([$path, $groupId], [ParameterType::STRING, ParameterType::INTEGER])));
		}

		$this->getDatabase()->setQuery($query)->execute();

		echo new JsonResponse($groups, $app->getLanguage()->_('PLG_FILESYSTEM_DPPERMISSIONS_TEXT_SUCCESSFULLY_SAVED_GROUPS'));

		$app->close();
	}

	public function getGroups(): void
	{
		Session::checkToken('get') or die(Text::_('JINVALID_TOKEN'));

		$app = $this->getApplication();
		if (!$app instanceof CMSApplicationInterface) {
			return;
		}

		if (!$this->canSetPermissions()) {
			throw new \Exception('Not allowed', 403);
		}

		$path = $app->getInput()->getString('path', '');
		if ($path === '' || $path === '0') {
			return;
		}

		$path = str_replace('//', '/', Path::clean($path, '/'));

		$query = $this->getDatabase()->getQuery(true);
		$query->select('group_id')->from('#__dppermissions')->where('path=:path')->bind(':path', $path);

		$this->getDatabase()->setQuery($query);

		$groups = array_map(static fn (array $group) => $group['group_id'], $this->getDatabase()->loadAssocList());

		echo new JsonResponse($groups);

		$app->close();
	}

	private function canSetPermissions(): bool
	{
		$user = $this->getApplication() instanceof CMSApplicationInterface ? $this->getApplication()->getIdentity() : null;
		if ($user === null) {
			return false;
		}

		// When global admin then return all to prevent lock out
		if ($user->authorise('core.admin')) {
			return true;
		}

		// Check if the current user groups are in the allowed groups
		$userGroups = Access::getGroupsByUser($user->id);
		return array_intersect($userGroups, $this->params->get('allowed_groups', [])) !== [];
	}
}
