<?php
/**
 * @package   DPMedia
 * @copyright Copyright (C) 2023 Digital Peak GmbH. <https://www.digital-peak.com>
 * @license   https://www.gnu.org/licenses/gpl-3.0.html GNU/GPL
 */

namespace DigitalPeak\Plugin\Filesystem\Ftp\Client;

use Exception;
use FtpClient\FtpClient;
use FtpClient\FtpException;
use Joomla\Registry\Registry;
use phpseclib3\Crypt\PublicKeyLoader;
use phpseclib3\Exception\FileNotFoundException;
use phpseclib3\Net\SFTP;

/**
 * Class to abstract the differences between the FTP and SFTP implementation.
 */
class Client
{
	private ?FtpClient $ftp = null;
	private ?SFTP $sftp     = null;

	/**
	 * Returns a list of files for the given path.
	 *
	 * @return array
	 * @throws \Exception
	 * @throws \UnexpectedValueException
	 */
	public function getFiles(string $path = '/'): array
	{
		if ($this->ftp instanceof FtpClient) {
			$files = $this->ftp->mlsd($path);

			// When the server doesn't support mlsd, then we need to do a fallback on normal list
			// @phpstan-ignore-next-line
			if ($files === false) {
				try {
					$files = $this->ftp->nlist($path);
				} catch (\Exception) {
					// Ignoring here the exception as we want the real error
				}
			}

			// @phpstan-ignore-next-line
			if ($files === false) {
				throw new \Exception(error_get_last() && stripos(error_get_last()['message'], 'ftp') !== false ? error_get_last()['message'] : 'Error');
			}

			foreach ($files as $key => $file) {
				// When nlist fetch
				if (\is_string($file)) {
					$entry       = basename($file);
					$entry       = ['name' => $entry, 'type' => strpos($entry, '.') ? 'file' : 'dir'];
					$files[$key] = $entry;
				}
			}

			return $files;
		}

		if ($this->sftp instanceof SFTP) {
			$files = $this->sftp->rawlist($path);
			if ($files === false) {
				throw new \Exception($this->sftp->getLastSFTPError() ?: 'Error');
			}

			foreach ($files as $key => $file) {
				$file['name']   = $file['filename'];
				$file['type']   = $file['type'] === 2 ? 'dir' : 'file';
				$file['modify'] = (new \DateTime('@' . $file['mtime']))->format('YmdHis');
				$files[$key]    = $file;
			}

			return $files;
		}

		return [];
	}

	/**
	 * Returns the downloaded file as string for the given path.
	 *
	 * @return string
	 * @throws \UnexpectedValueException
	 */
	public function download(string $path = '/'): string
	{
		if ($this->ftp instanceof FtpClient) {
			$handle = fopen('php://temp', 'w+');
			if ($handle === false) {
				throw new \Exception('Can not open file!');
			}

			$this->ftp->fget($handle, $path, FTP_BINARY, 0);
			rewind($handle);
			return stream_get_contents($handle) ?: '';
		}

		if ($this->sftp instanceof SFTP) {
			$content = $this->sftp->get($path);
			if ($content === false) {
				throw new \Exception($this->sftp->getLastSFTPError() ?: 'Error');
			}

			return \is_string($content) ? $content : '';
		}

		return '';
	}

	/**
	 * Upload the given data on the given path.
	 *
	 * @return void
	 * @throws FtpException
	 * @throws \UnexpectedValueException
	 * @throws \BadFunctionCallException
	 * @throws FileNotFoundException
	 * @throws \Exception
	 */
	public function upload(string $path, mixed $data): void
	{
		if ($this->ftp instanceof FtpClient) {
			$this->ftp->putFromString($path, $data);
		}

		if ($this->sftp instanceof SFTP) {
			$result = $this->sftp->put($path, $data);
			if ($result === false) {
				throw new \Exception($this->sftp->getLastSFTPError() ?: 'Error');
			}
		}
	}

	/**
	 * Deletes the file or folder on the given path.
	 *
	 * @return void
	 * @throws \Exception
	 * @throws \UnexpectedValueException
	 */
	public function delete(string $path): void
	{
		if ($this->ftp instanceof FtpClient) {
			$success = $this->ftp->remove($path, true);
			if (!$success) {
				throw new \Exception(error_get_last() !== null && error_get_last() !== [] ? error_get_last()['message'] : 'Error');
			}
		}

		if ($this->sftp instanceof SFTP) {
			$result = $this->sftp->delete($path);
			if ($result === false) {
				throw new \Exception($this->sftp->getLastSFTPError() ?: 'Error');
			}
		}
	}

	/**
	 * Renames the source file or folder to the given destination.
	 *
	 * @return void
	 * @throws \Exception
	 * @throws \UnexpectedValueException
	 */
	public function rename(string $sourcePath, string $destinationPath): void
	{
		if ($this->ftp instanceof FtpClient) {
			$success = $this->ftp->rename($sourcePath, $destinationPath);
			if (!$success) {
				throw new \Exception(error_get_last() !== null && error_get_last() !== [] ? error_get_last()['message'] : 'Error');
			}
		}

		if ($this->sftp instanceof SFTP) {
			$result = $this->sftp->rename($sourcePath, $destinationPath);
			if ($result === false) {
				throw new \Exception($this->sftp->getLastSFTPError() ?: 'Error');
			}
		}
	}

	/**
	 * Creates the folder on the given path.
	 *
	 * @return string
	 * @throws FtpException
	 * @throws \UnexpectedValueException
	 * @throws \InvalidArgumentException
	 * @throws \RuntimeException
	 * @throws \LengthException
	 * @throws \Exception
	 */
	public function mkdir(string $path): string
	{
		$newPath = '';
		if ($this->ftp instanceof FtpClient) {
			$newPath = $this->ftp->mkdir($path);
			if ($newPath === false) {
				$lastError = error_get_last();
				throw new \Exception($lastError !== null && $lastError !== [] ? $lastError['message'] : 'Error');
			}
		}

		if ($this->sftp instanceof SFTP) {
			$newPath = $this->sftp->mkdir($path);
			if ($newPath === false) {
				throw new \Exception($this->sftp->getLastSFTPError() ?: 'Error');
			}
		}

		return $path;
	}

	/**
	 * Connects the client for the given config.
	 */
	public function connect(Registry $config): void
	{
		if ($this->ftp instanceof FtpClient || $this->sftp instanceof SFTP) {
			return;
		}

		if ($config->get('protocol', 'ftp') === 'ftp') {
			$this->ftp = new FtpClient();
			$this->ftp->connect($config->get('host'), $config->get('ssl', 1), $config->get('port', 21));
			$this->ftp->login($config->get('username'), $config->get('password'));
			$this->ftp->pasv((int)$config->get('pasv', 0) === 1);
		}

		if ($config->get('protocol', 'ftp') === 'sftp') {
			$this->sftp = new SFTP($config->get('host'), $config->get('port', 21));

			$password = $config->get('password');
			if (!$password && file_exists($config->get('key_file', ''))) {
				$password = PublicKeyLoader::load(file_get_contents($config->get('key_file')) ?: '');
			}

			$this->sftp->login($config->get('username'), $password);
			if ($this->sftp->getLastSFTPError() !== '' && $this->sftp->getLastSFTPError() !== '0') {
				throw new \Exception($this->sftp->getLastSFTPError());
			}
		}
	}
}
